<?php

/**
 * Get job status by passed task id
 *
 * @param  mixed  $id  task id
 *
 * @return array
 */
function get_job_status_by_id($id)
{
	$CI       = &get_instance();
	$statuses = $CI->jobs_model->get_statuses();
	foreach ($statuses as $s)
	{
		if ($s['id'] == $id)
		{
			$status = $s;

			break;
		}
	}

	return $status;
}

/**
 * Return jobs summary formated data
 *
 * @param  null  $rel_id
 * @param  null  $rel_type
 *
 * @return array
 */
function jobs_summary_data(): array
{
	$CI           = &get_instance();
	$jobs_summary = [];
	$statuses     = $CI->jobs_model->get_statuses();
	foreach ($statuses as $status)
	{
		$jobs_where = 'status = '.$CI->db->escape_str($status['id']);
		if (staff_cant('view', 'tasks'))
		{
			$jobs_where .= ' '.get_jobs_where_string();
		}
		$jobs_my_where = 'jobid IN(SELECT job_id FROM '.db_prefix().'tjm_job_specialists WHERE staff_id='.get_staff_user_id().') AND status='.$CI->db->escape_str($status['id']);

		$summary                  = [];
		$summary['total_jobs']    = total_rows(db_prefix().'tjm_jobs', $jobs_where);
		$summary['total_my_jobs'] = total_rows(db_prefix().'tjm_jobs', $jobs_my_where);
		$summary['color']         = $status['color'];
		$summary['name']          = $status['name'];
		$summary['status_id']     = $status['id'];
		$jobs_summary[]           = $summary;
	}

	return $jobs_summary;
}

/**
 * Function that format task status for the final user
 *
 * @param  string  $id  status id
 * @param  boolean  $text
 * @param  boolean  $clean
 *
 * @return string
 */
function format_job_status($status, $text = FALSE, $clean = FALSE)
{
	if ( ! is_array($status))
	{
		$status = get_job_status_by_id($status);
	}

	$status_name = $status['name'];

	$status_name = e(hooks()->apply_filters('task_status_name', $status_name, $status));

	if ($clean == TRUE)
	{
		return $status_name;
	}

	$style = '';
	$class = '';
	if ($text == FALSE)
	{
		$style = 'color:'.$status['color'].';border:1px solid '.adjust_hex_brightness($status['color'], 0.4).';background: '.adjust_hex_brightness($status['color'], 0.04).';';
		$class = 'label';
	} else
	{
		$style = 'color:'.$status['color'].';';
	}

	return '<span class="'.$class.'" style="'.$style.'">'.$status_name.'</span>';
}

/**
 * Tasks html table used all over the application for relation tasks
 * This table is not used for the main tasks table
 *
 * @param  array  $table_attributes
 *
 * @return string
 */
function init_relation_jobs_table($table_attributes = [], $filtersWrapperId = 'vueApp')
{
	$table_data = [
		_l('the_number_sign'),
		_l('job_name'),
		_l('job_status'),
		_l('job_date_start'),
		[
			'name'     => _l('job_due_date'),
			'th_attrs' => ['class' => 'duedate'],
		],
		_l('job_assigned'),
		_l('job_priority'),
	];

	array_unshift($table_data, [
		'name'     => '<span class="hide"> - </span><div class="checkbox mass_select_all_wrap"><input type="checkbox" id="mass_select_all" data-to-table="jobs"><label></label></div>',
		'th_attrs' => ['class' => isset($bulk_actions) ? '' : 'not_visible'],
	]);

	$name = 'rel-tasks';
	if ($table_attributes['data-new-rel-type'] == 'lead')
	{
		$name = 'rel-tasks-leads';
	}

	$tasks_table = App_table::find('related_tasks');

	$table      = '';
	$CI         = &get_instance();
	$table_name = '.table-'.$name;

	$CI->load->view('admin/tasks/filters', [
		'tasks_table'        => $tasks_table,
		'filters_wrapper_id' => $filtersWrapperId,
	]);

	if (staff_can('create', 'tasks'))
	{
		$disabled   = '';
		$table_name = addslashes($table_name);
		if ($table_attributes['data-new-rel-type'] == 'customer' && is_numeric($table_attributes['data-new-rel-id']))
		{
			if (
				total_rows(db_prefix().'clients', [
					'active' => 0,
					'userid' => $table_attributes['data-new-rel-id'],
				]) > 0
			)
			{
				$disabled = ' disabled';
			}
		}
		// projects have button on top
		if ($table_attributes['data-new-rel-type'] != 'project')
		{
			echo
				"<a href='#' class='btn btn-primary pull-left mright5 new-task-relation"
				.$disabled
				."' onclick=\"new_task_from_relation('$table_name'); return false;\" data-rel-id='"
				.$table_attributes['data-new-rel-id']
				."' data-rel-type='"
				.$table_attributes['data-new-rel-type']
				."'><i class=\"fa-regular fa-plus tw-mr-1\"></i>"
				._l('new_task')
				.'</a>';
		}
	}

	if ($table_attributes['data-new-rel-type'] == 'project')
	{
		echo
			"<a href='"
			.admin_url('tasks/list_tasks?project_id='.$table_attributes['data-new-rel-id'].'&kanban=true')
			."' class='btn btn-default mright5 mbot15 hidden-xs' data-toggle='tooltip' data-title='"
			._l('view_kanban')
			."' data-placement='top'><i class='fa-solid fa-grip-vertical'></i></a>";
		echo "<a href='".admin_url('tasks/detailed_overview?project_id='.$table_attributes['data-new-rel-id'])."' class='btn btn-success pull-rigsht mbot15'>"._l('detailed_overview').'</a>';
		echo '<div class="clearfix"></div>';
		echo $CI->load->view('admin/tasks/_bulk_actions', ['table' => '.table-rel-tasks'], TRUE);
		echo $CI->load->view('admin/tasks/_summary', ['rel_id' => $table_attributes['data-new-rel-id'], 'rel_type' => 'project', 'table' => $table_name], TRUE);
		echo '<a href="#" data-toggle="modal" data-target="#tasks_bulk_actions" class="hide bulk-actions-btn table-btn" data-table=".table-rel-tasks">'._l('bulk_actions').'</a>';
	} elseif ($table_attributes['data-new-rel-type'] == 'customer')
	{
		echo '<div class="clearfix"></div>';
		echo '<div id="tasks_related_filter" class="mtop15">';
		echo '<p class="bold">'._l('task_related_to').': </p>';

		echo
			'<div class="checkbox checkbox-inline">
        <input type="checkbox" checked value="customer" disabled id="ts_rel_to_customer" name="tasks_related_to[]">
        <label for="ts_rel_to_customer">'
			._l('client')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="project" id="ts_rel_to_project" name="tasks_related_to[]">
        <label for="ts_rel_to_project">'
			._l('projects')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="invoice" id="ts_rel_to_invoice" name="tasks_related_to[]">
        <label for="ts_rel_to_invoice">'
			._l('invoices')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="estimate" id="ts_rel_to_estimate" name="tasks_related_to[]">
        <label for="ts_rel_to_estimate">'
			._l('estimates')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="contract" id="ts_rel_to_contract" name="tasks_related_to[]">
        <label for="ts_rel_to_contract">'
			._l('contracts')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="ticket" id="ts_rel_to_ticket" name="tasks_related_to[]">
        <label for="ts_rel_to_ticket">'
			._l('tickets')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="expense" id="ts_rel_to_expense" name="tasks_related_to[]">
        <label for="ts_rel_to_expense">'
			._l('expenses')
			.'</label>
        </div>

        <div class="checkbox checkbox-inline">
        <input type="checkbox" value="proposal" id="ts_rel_to_proposal" name="tasks_related_to[]">
        <label for="ts_rel_to_proposal">'
			._l('proposals')
			.'</label>
        </div>';
		echo form_hidden('tasks_related_to');
		echo '</div>';
	}
	echo "<div class='clearfix'></div>";

	// If new column is added on tasks relations table this will not work fine
	// In this case we need to add new identifier eq task-relation
	$table_attributes['data-last-order-identifier'] = 'tasks';
	$table_attributes['data-default-order']         = get_table_last_order('tasks');
	if ($table_attributes['data-new-rel-type'] != 'project')
	{
		echo '<hr />';
	}
	$table_attributes['id'] = 'related_tasks';

	$table .= render_datatable($table_data, $name, ['number-index-1'], $table_attributes);

	return $table;
}

/**
 * Get predefined property types
 *
 * @return array
 */
function get_property_types(): array
{
	return [
		'Apartment'      => 'Apartment',
		'Converted flat' => 'Converted flat',
		'Studio flat'    => 'Studio flat',
		'Cottage'        => 'Cottage',
		'Bungalow'       => 'Bungalow',
		'Detached house' => 'Detached house',
		'Terraced house' => 'Terraced house',
		'End of terrace' => 'End of terrace',
		'Maisonette'     => 'Maisonette',
		'Commercial'     => 'Commercial',
		'Other'          => 'Other',
	];
}

/**
 * Return predefined job types
 *
 * @return array
 */
function get_job_types()
{
	return [
		[
			'id'    => 1,
			'name'  => _l('job_type_building_and_construction'),
			'color' => '#777',
		],
		[
			'id'    => 2,
			'name'  => _l('job_type_carpentry'),
			'color' => '#03a9f4',
		],
		[
			'id'    => 3,
			'name'  => _l('job_type_electrical'),
			'color' => '#ff6f00',
		],
		[
			'id'    => 4,
			'name'  => _l('job_type_flooring'),
			'color' => '#fc2d42',
		],
		[
			'id'    => 5,
			'name'  => _l('job_type_gardening'),
			'color' => '#777',
		],
		[
			'id'    => 6,
			'name'  => _l('job_type_handyman_services'),
			'color' => '#03a9f4',
		],
		[
			'id'    => 7,
			'name'  => _l('job_type_hvac_services'),
			'color' => '#ff6f00',
		],
		[
			'id'    => 8,
			'name'  => _l('job_type_locksmith_services'),
			'color' => '#fc2d42',
		],
		[
			'id'    => 9,
			'name'  => _l('job_type_painting_and_decorating'),
			'color' => '#777',
		],
		[
			'id'    => 10,
			'name'  => _l('job_type_plumbing'),
			'color' => '#03a9f4',
		],
		[
			'id'    => 11,
			'name'  => _l('job_type_roofing'),
			'color' => '#ff6f00',
		],
		[
			'id'    => 12,
			'name'  => _l('job_type_tiling'),
			'color' => '#fc2d42',
		],
		[
			'id'    => 12,
			'name'  => _l('job_type_window_fitting'),
			'color' => '#fc2d42',
		],
	];
}

/**
 * @param  string  $job_type_name
 *
 * @return int|null
 */
function get_job_type_id_by_name(string $job_type_name): null|int
{
	$job_types = get_job_types();

	$job_type = array_filter($job_types, function ($job_type) use ($job_type_name) {
		return $job_type['name'] === $job_type_name;
	});

	if (count($job_type) > 0)
	{
		return array_shift($job_type)['id'];
	}

	return NULL;
}

/**
 * Return predefined job types
 *
 * @return array
 */
function get_job_property_types()
{
	return [
		[
			'id'    => 'Flat',
			'name'  => _l('Flat'),
			'color' => '#777',
		],
		[
			'id'    => 'Terraced-house',
			'name'  => _l('Terraced-house'),
			'color' => '#03a9f4',
		],
		[
			'id'    => 'Other',
			'name'  => _l('Other'),
			'color' => '#ff6f00',
		],
	];
}

/**
 * Format job type based on passed job type id
 *
 * @param  mixed  $id
 *
 * @return string
 */
function job_type($id)
{
	foreach (get_job_types() as $type)
	{
		if ($type['id'] == $id)
		{
			return $type['name'];
		}
	}

	// Not exists?
	return $id;
}

function get_landlord_profile_tabs()
{
	return [
		'profile'    => [
			'slug'     => 'profile',
			'name'     => 'Profile',
			'icon'     => 'profile',
			'view'     => 'profile',
			'position' => 'profile',
			'href'     => 'profile',
		],
		'properties' => [
			'slug'     => 'properties',
			'name'     => 'Properties',
			'icon'     => 'properties',
			'view'     => 'properties',
			'position' => '10',
			'href'     => 'properties',
		],
	];
}

function send_scheduled_time_email(object $job): bool
{
	if ( ! isset($job->property) || ! isset($job->landlord))
	{
		return FALSE;
	}
	$landlord = $job->landlord;
	$property = $job->property;

	$CI = &get_instance();
	$CI->load->config('email');

	$CI->email->clear(TRUE);
	$CI->email->set_newline(config_item('newline'));

	$from_email = get_option('smtp_email');
	$from_name  = get_option('companyname');
	$CI->email->from($from_email, $from_name);

	$CI->email->subject('Job scheduled');

	$date_visit = $job->preferred_visit_date;
	$date_visit = Datetime::createFromFormat('Y-m-d H:i:s', $date_visit);
	$formatted  = $date_visit->format('d F Y, h:ia');

	$body = 'Dear '.$landlord['first_name'].' '.$landlord['last_name'].','.PHP_EOL.PHP_EOL;
	$body .= 'The following job has been scheduled for your property.'.PHP_EOL.PHP_EOL;
	$body .= 'Job Title: <strong>'.$job->job_title.'</strong>'.PHP_EOL;
	$body .= 'Property Address: <strong>'.$property['property_address'].'</strong>'.PHP_EOL;
	$body .= 'Visit time: <strong>'.$formatted.'</strong>'.PHP_EOL.PHP_EOL;
	$body .= 'If you have any questions please contact us as soon as possible.'.PHP_EOL.PHP_EOL;
	$body .= 'Sincerely,'.PHP_EOL.PHP_EOL;
	$body .= get_option('companyname');

	$body = str_replace(PHP_EOL, '<br>', $body);
	$CI->email->message($body);
	$CI->email->to($landlord['email']);

	if ($CI->email->send())
	{
		return TRUE;
	}

	return FALSE;
}

function send_job_completed_email(object $job): bool
{
	if ( ! isset($job->property) || ! isset($job->landlord))
	{
		return FALSE;
	}
	$landlord = $job->landlord;
	$property = $job->property;

	$CI = &get_instance();
	$CI->load->config('email');

	$CI->email->clear(TRUE);
	$CI->email->set_newline(config_item('newline'));

	$from_email = get_option('smtp_email');
	$from_name  = get_option('companyname');
	$CI->email->from($from_email, $from_name);

	$CI->email->subject('Job completed');

	$body = 'Dear '.$landlord['first_name'].' '.$landlord['last_name'].','.PHP_EOL.PHP_EOL;
	$body .= 'The following job has been completed at your property.'.PHP_EOL.PHP_EOL;
	$body .= 'Job Title: <strong>'.$job->job_title.'</strong>'.PHP_EOL;
	$body .= 'Property: <strong>'.$property['property_name'].'</strong>'.PHP_EOL.PHP_EOL;
	$body .= 'If you have any questions or concerns, please contact us.'.PHP_EOL.PHP_EOL;
	$body .= 'You will receive a separate email containing an invoice for this work.'.PHP_EOL.PHP_EOL;
	$body .= 'Sincerely,'.PHP_EOL.PHP_EOL;
	$body .= get_option('companyname');

	$body = str_replace(PHP_EOL, '<br>', $body);
	$CI->email->message($body);
	$CI->email->to($landlord['email']);

	if ($CI->email->send())
	{
		return TRUE;
	}

	return FALSE;
}

function send_job_reported_email(object $job): bool
{
	if ( ! isset($job->property) || ! isset($job->landlord))
	{
		return FALSE;
	}
	$landlord = $job->landlord;

	$CI = &get_instance();
	$CI->load->config('email');

	$CI->email->clear(TRUE);
	$CI->email->set_newline(config_item('newline'));

	$from_email = get_option('smtp_email');
	$from_name  = get_option('companyname');
	$CI->email->from($from_email, $from_name);

	$CI->email->subject('Job reported');

	$body = 'Dear '.$landlord['first_name'].' '.$landlord['last_name'].','.PHP_EOL.PHP_EOL;
	$body .= 'You have reported the job: '.'<strong>'.$job->job_title.'</strong>. ';
	$body .= 'This is to inform you that we are in the process of scheduling this job. Once a specialist has been assigned to this job and a visit has been scheduled, you will be notified.'.PHP_EOL;
	$body .= 'Sincerely,'.PHP_EOL.PHP_EOL;
	$body .= get_option('companyname');

	$body = str_replace(PHP_EOL, '<br>', $body);
	$CI->email->message($body);
	$CI->email->to($landlord['email']);

	if ($CI->email->send())
	{
		return TRUE;
	}

	return FALSE;
}

if ( ! function_exists('tjm_get_all_api_clients'))
{
	/**
	 * @return array
	 */
	function tjm_get_all_api_clients(): array
	{
		$CI = &get_instance();
		$CI->db->select(['*']);
		$CI->db->from(db_prefix().'tjm_api_clients');

		return $CI->db->get()->result_array();
	}
}

if ( ! function_exists('tjm_create_slug'))
{
	function tjm_create_slug($text): string
	{
		// Replace non-alphanumeric characters with hyphens
		$text = preg_replace('/[^a-zA-Z0-9]/', '-', $text);

		// Convert to lowercase
		$text = strtolower($text);

		// Remove duplicate hyphens
		$text = preg_replace('/-+/', '-', $text);

		// Remove hyphens from beginning and end
		return trim($text, '-');
	}
}

if ( ! function_exists('tjm_handle_property_attachments_array'))
{
	function tjm_handle_property_attachments_array($id, $index_name = 'attachments'): bool|array
	{
		$uploaded_files = [];
		$path           = get_upload_path_by_type('property').$id.'/';
		if ( ! file_exists($path))
		{
			mkdir($path, 0755, TRUE);
			fopen(rtrim($path, '/').'/'.'index.html', 'w');
		}

		if (isset($_FILES[$index_name]['name']) && ($_FILES[$index_name]['name'] != '' || is_array($_FILES[$index_name]['name']) && count($_FILES[$index_name]['name']) > 0))
		{
			if ( ! is_array($_FILES[$index_name]['name']))
			{
				$_FILES[$index_name]['name']     = [$_FILES[$index_name]['name']];
				$_FILES[$index_name]['type']     = [$_FILES[$index_name]['type']];
				$_FILES[$index_name]['tmp_name'] = [$_FILES[$index_name]['tmp_name']];
				$_FILES[$index_name]['error']    = [$_FILES[$index_name]['error']];
				$_FILES[$index_name]['size']     = [$_FILES[$index_name]['size']];
			}

			_file_attachments_index_fix($index_name);
			for ($i = 0; $i < count($_FILES[$index_name]['name']); $i++)
			{
				// Get the temp file path
				$tmpFilePath = $_FILES[$index_name]['tmp_name'][$i];

				// Make sure we have a filepath
				if ( ! empty($tmpFilePath) && $tmpFilePath != '')
				{
					if (_perfex_upload_error($_FILES[$index_name]['error'][$i]) || ! _upload_extension_allowed($_FILES[$index_name]['name'][$i]))
					{
						continue;
					}

					$filename    = unique_filename($path, $_FILES[$index_name]['name'][$i]);
					$newFilePath = $path.$filename;

					if (move_uploaded_file($tmpFilePath, $newFilePath))
					{
						array_push($uploaded_files, [
							'file_name' => $filename,
							'filetype'  => $_FILES[$index_name]['type'][$i],
						]);

						if (is_image($newFilePath))
						{
							create_img_thumb($path, $filename);
						}
					}
				}
			}
		}

		if (count($uploaded_files) > 0)
		{
			return $uploaded_files;
		}

		return FALSE;
	}
}

if ( ! function_exists('tjm_force_download'))
{
	function tjm_force_download($filename = '', $data = '', $set_mime = FALSE)
	{
		if ($filename === '' or $data === '')
		{
			return;
		} elseif ($data === NULL)
		{
			if ( ! @is_file($filename) or ($filesize = @filesize($filename)) === FALSE)
			{
				return;
			}

			$filepath = $filename;
			$filename = explode('/', str_replace(DIRECTORY_SEPARATOR, '/', $filename));
			$filename = end($filename);
		} else
		{
			$filesize = strlen($data);
		}

		// Set the default MIME type to send
		$mime = 'application/octet-stream';

		$x         = explode('.', $filename);
		$extension = end($x);

		if ($set_mime === TRUE)
		{
			if (count($x) === 1 or $extension === '')
			{
				/* If we're going to detect the MIME type,
				 * we'll need a file extension.
				 */
				return;
			}

			// Load the mime types
			$mimes = &get_mimes();

			// Only change the default MIME if we can find one
			if (isset($mimes[$extension]))
			{
				$mime = is_array($mimes[$extension]) ? $mimes[$extension][0] : $mimes[$extension];
			}
		}

		/* It was reported that browsers on Android 2.1 (and possibly older as well)
		 * need to have the filename extension upper-cased in order to be able to
		 * download it.
		 *
		 * Reference: http://digiblog.de/2011/04/19/android-and-the-download-file-headers/
		 */
		if (count($x) !== 1 && isset($_SERVER['HTTP_USER_AGENT']) && preg_match('/Android\s(1|2\.[01])/', $_SERVER['HTTP_USER_AGENT']))
		{
			$x[count($x) - 1] = strtoupper($extension);
			$filename         = implode('.', $x);
		}

		if ($data === NULL && ($fp = @fopen($filepath, 'rb')) === FALSE)
		{
			return;
		}

		// Clean output buffer
		if (ob_get_level() !== 0 && @ob_end_clean() === FALSE)
		{
			@ob_clean();
		}

		// Generate the server headers
		header('Content-Type: '.$mime);
		header('Content-Disposition: attachment; filename="'.$filename.'"');
		header('Expires: 0');
		header('Content-Transfer-Encoding: binary');
		header('Content-Length: '.$filesize);
		header('Cache-Control: private, no-transform, no-store, must-revalidate');

		// If we have raw data - just dump it
		if ($data !== NULL)
		{
			exit($data);
		}

		// Flush 1MB chunks of data
		while ( ! feof($fp) && ($data = fread($fp, 1048576)) !== FALSE)
		{
			echo $data;
		}

		fclose($fp);
		exit();
	}
}

function tjm_debug_var(mixed $var, bool $die = TRUE)
{
	echo '<pre style="padding: 1rem 2rem;background-color: grey; color: white;">';
	print_r($var);
	echo '</pre>';
	if ($die)
	{
		die();
	}
}

/**
 * Check job restrictions - hash, clientid
 *
 * @param  mixed  $id  estimate id
 * @param  string  $hash  estimate hash
 */
function check_job_restrictions($id = '', $hash = ''): void
{
	$path = APP_MODULES_PATH.TRADE_JOB_MANAGEMENT_MODULE_NAME.'/models/Jobs_model.php';

	require_once $path;
	$CI = &get_instance();
	$CI->load->model('jobs_model');
	if (empty($hash) || empty($id))
	{
		show_404();
	}
	$job = $CI->jobs_model->get($id);
	if ( ! $job || ($job->job_hash != $hash))
	{
		show_404();
	}
	// Do one more check
	if (is_client_logged_in())
	{
		if ($job->clientid != get_client_user_id())
		{
			show_404();
		}
	}
}

/**
 * Used for job sign info array
 *
 * @param  boolean  $empty  should the array values be empty or taken from $_POST
 *
 * @return array
 */
function get_job_sign_info_array($empty = FALSE)
{
	$CI        = &get_instance();
	$signature = NULL;

	if (isset($GLOBALS['processed_digital_signature']))
	{
		$signature = $GLOBALS['processed_digital_signature'];
		unset($GLOBALS['processed_digital_signature']);
	}

	$data = [
		'signature'          => $signature,
		'job_sign_firstname' => ! $empty ? $CI->input->post('job_sign_firstname') : NULL,
		'job_sign_lastname'  => ! $empty ? $CI->input->post('job_sign_lastname') : NULL,
		'job_sign_email'     => ! $empty ? $CI->input->post('job_sign_email') : NULL,
		'job_sign_date'      => ! $empty ? date('Y-m-d H:i:s') : NULL,
		'job_sign_ip'        => ! $empty ? $CI->input->ip_address() : NULL,
	];

	return hooks()->apply_filters('job_sign_info_array', $data, $empty);
}

function generate_job_hash(array $job): string
{
	$job_hash_raw = sprintf('%s%s%s%s', $job['jobid'], $job['job_type_id'], $job['property_id'], $job['job_type_id']);

	return md5($job_hash_raw);
}

/**
 * Handle daily log file attachments
 *
 * @param  int  $job_id  job id
 * @param  int  $daily_log_id  daily log id
 *
 * @return array
 */
function handle_job_daily_log_attachments($job_id, $daily_log_id)
{
	$path          = get_upload_path_by_type('job').$job_id.'/daily_logs/';
	$uploadedFiles = [];
	$totalUploaded = 0;

	if (isset($_FILES['file']['name']) && is_array($_FILES['file']['name']) && count($_FILES['file']['name']) > 0)
	{
		if ( ! is_dir($path))
		{
			mkdir($path, 0755, TRUE);
		}

		$allowed_extensions = get_option('ticket_attachments_file_extensions');

		for ($i = 0; $i < count($_FILES['file']['name']); $i++)
		{
			// Skip empty file inputs
			if ($_FILES['file']['name'][$i] == '')
			{
				continue;
			}

			// Do not rename because we want to show the original name in the UI
			$_FILES['file']['name'][$i] = unique_filename($path, $_FILES['file']['name'][$i]);

			$_FILES['file']['name'][$i] = str_replace(' ', '_', $_FILES['file']['name'][$i]);

			if ( ! file_exists($path))
			{
				mkdir($path, 0755, TRUE);
			}

			$filename = unique_filename($path, $_FILES['file']['name'][$i]);

			// Check file extension
			$extension      = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
			$allowed        = explode(',', $allowed_extensions);
			$allowed        = array_map('trim', $allowed);
			$notAllowedText = '';
			if ( ! in_array('.'.$extension, $allowed))
			{
				$notAllowedText = 'File type not allowed';
			}

			if ($notAllowedText === '')
			{
				$uploadedFiles[$i]              = [];
				$uploadedFiles[$i]['file_name'] = $filename;
				$uploadedFiles[$i]['filetype']  = $_FILES['file']['type'][$i];
			}

			if ($notAllowedText !== '')
			{
				header('HTTP/1.0 400 Bad error');
				echo $notAllowedText;
				die;
			}

			$total_errors = 0;
			if ( ! isset($uploadedFiles[$i]))
			{
				continue;
			}

			if ( ! move_uploaded_file($_FILES['file']['tmp_name'][$i], $path.$filename))
			{
				$total_errors++;
			} else
			{
				$totalUploaded++;
			}
		}
	}

	return $uploadedFiles;
}

/**
 * Get upload path for job daily logs
 *
 * @param  int  $job_id  job id
 *
 * @return string
 */
function get_job_daily_log_upload_path($job_id)
{
	return get_upload_path_by_type('job').$job_id.'/daily_logs/';
}
