<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Apis extends ClientsController {

	public function __construct()
	{
		parent::__construct();
		$this->load->model('jobs_model');
		$this->load->model('job_types_model');
		$this->load->model('properties_model');
		$this->load->model('property_types_model');
		$this->load->model('landlords_model');
		$this->load->model('clients_model');
	}

	protected const BASE_PATH = '/v1';

	protected const MODULE_NAME = 'project_management_enhancements';

	protected const PASSWORD_PREFIX = 'landlords_';

	/**
	 * API /apis/jobs
	 *
	 * @return void
	 */
	public function jobs()
	{
		$data = json_decode(file_get_contents('php://input'), TRUE);
		if (empty($data) || json_last_error() !== JSON_ERROR_NONE)
		{
			header('Content-Type: application/json; charset=utf-8');
			header('HTTP/1.1 405 Method Not Allowed');
			http_response_code(405);

			echo
			json_encode([
				'success' => FALSE,
				'error'   => 'Empty data',
			]);
			exit();
		}

		$validated_client = $this->validate_request_client();
		if (isset($validated_client['error']))
		{
			header('Content-Type: application/json; charset=utf-8');
			header('HTTP/1.1 403 Forbidden');
			http_response_code($validated_client['code'] ?? 403);

			echo
			json_encode([
				'success'    => FALSE,
				'error'      => $validated_client['error'],
				'error_code' => $validated_client['error_code'],
				'code'       => 403,
			]);
			exit();
		}

		$valid_ip = $this->validate_request_ip($validated_client);
		if ($valid_ip !== TRUE)
		{
			header('Content-Type: application/json; charset=utf-8');
			header('HTTP/1.1 403 Forbidden');
			http_response_code($validated_client['code'] ?? 403);

			echo json_encode(array_merge([
				'success' => FALSE,
			], $valid_ip));

			exit();
		}

		$validated_data = $this->validate_request_data($data);
		if (isset($validated_data['error']))
		{
			header('Content-Type: application/json; charset=utf-8');
			header('HTTP/1.1 400 Bad Request');
			http_response_code($validated_data['code'] ?? 400);

			echo json_encode(array_merge([
				'success' => FALSE,
			], $validated_data));

			exit();
		}

		$request_landlord = $validated_data['landlord'];

		$landlord = $this->get_landlord($request_landlord['contact_email']);
		if ( ! $landlord)
		{
			$landlord = $this->create_landlord($request_landlord);
		}

		$job = $this->create_job($landlord, $validated_data['job']);

		header('Content-Type: application/json; charset=utf-8');
		header('HTTP/1.1 200 Ok');
		http_response_code(200);

		echo
		json_encode([
			'success' => TRUE,
			'data'    => [
				'landlord' => $landlord,
				'job'      => $job,
			],
		]);
		exit();
	}

	private function create_job($landlord, $job_data)
	{
		$CI = &get_instance();

		$property_name = $job_data['property_name'];
		$property      = $this->get_property($landlord, $property_name);
		if ( ! $property)
		{
			$property = $this->create_property($landlord, $property_name);
		}

		$job_type = $this->job_types_model->get_by_name($job_data['job_type']);
		$data     = [
			'job_title'            => $job_data['job_name'],
			'job_type_id'          => $job_type->jobtypeid,
			'priority'             => $job_data['job_priority'],
			'property_id'          => $property['id'],
			'preferred_visit_date' => $job_data['visit_date'],
			'contact_name'         => $job_data['contact_name'],
			'contact_phone'        => $job_data['contact_phone_number'],
			'job_description'      => $job_data['description'] ?? '',
			'billed'               => 0,
			'status'               => 1,
		];

		$admin = $CI->db->select('staffid')->where('admin', 1)->get(db_prefix().'staff')->row_array();

		$data['specialists'] = $admin['staffid'];

		$id = $this->jobs_model->add($data);

		if ($id)
		{
			$job = $this->jobs_model->get($id);
			send_job_reported_email($job);

			return (array)$job;
		}

		return FALSE;
	}

	/**
	 * @param $contact_email
	 *
	 * @return object|null
	 */
	private function get_landlord($contact_email)
	{
		$CI = &get_instance();
		$CI->load->model('landlords_model');

		return $CI->landlords_model->get_landlord_by_email($contact_email);
	}

	/**
	 * Create new landlord
	 *
	 * @param  array  $data
	 *
	 * @return array | false
	 */
	private function create_landlord(array $data): array|false
	{
		if (isset($data['country']))
		{
			$all_countries = get_all_countries();
			$country       = array_filter($all_countries, function ($country) use ($data) {
				return $country['short_name'] === $data['country'];
			});

			$country         = array_shift($country);
			$data['country'] = $country['country_id'];
		}

		$landlord_email = $data['contact_email'] ?? '';
		$first_name     = $data['firstname'] ?? '';
		$last_name      = $data['lastname'] ?? '';
		$phone_number   = $data['phonenumber'] ?? '';

		$data_insert = [
			'first_name'   => $first_name,
			'last_name'    => $last_name,
			'phone_number' => $phone_number,
			'email'        => $landlord_email,
			'website'      => $data['website'] ?? '',
			'address'      => $data['address'] ?? '',
			'city'         => $data['city'] ?? '',
			'state'        => $data['state'] ?? '',
			'zip_code'     => $data['zip'] ?? '',
			'country'      => $data['country'] ?? '',
			'created_by'   => NULL,
		];

		$landlord = $this->landlords_model->get_landlord_by_email($landlord_email);
		if ($landlord)
		{
			return $landlord;
		}
		$id = $this->landlords_model->add($data_insert);

		if ($id)
		{
			$data['company']         = $first_name.' '.$last_name;
			$data['contact_email']   = $landlord_email;
			$data['phonenumber']     = $phone_number;
			$data['billing_street']  = $data['address'];
			$data['billing_city']    = $data['city'];
			$data['billing_state']   = $data['state'];
			$data['billing_zip']     = $data['zip_code'];
			$data['billing_country'] = $data['country'];

			$data['shipping_street']  = $data['address'];
			$data['shipping_city']    = $data['city'];
			$data['shipping_state']   = $data['state'];
			$data['shipping_zip']     = $data['zip_code'];
			$data['shipping_country'] = $data['country'];

			$contact_data = [
				'firstname'       => $first_name,
				'lastname'        => $last_name,
				'email'           => $landlord_email,
				'phonenumber'     => $phone_number,
				'password'        => static::PASSWORD_PREFIX.rand(0, 1000000),
				'is_primary'      => TRUE,
				'invoice_emails'  => TRUE,
				'estimate_emails' => TRUE,
				'task_emails'     => TRUE,
				'project_emails'  => TRUE,
			];

			unset($data['first_name']);
			unset($data['last_name']);
			unset($data['phone_number']);
			unset($data['email']);
			unset($data['zip_code']);

			$client_id = $this->clients_model->add($data);
			$this->clients_model->add_contact($contact_data, $client_id);

			$this->landlords_model->update_client_id($id, $client_id);

			return $this->get_landlord($landlord_email);
		}

		return FALSE;
	}

	/**
	 * API /apis/properties
	 */
	public function properties()
	{
		$validated_client = $this->validate_request_client();
		if (isset($validated_client['error']))
		{
			header('Content-Type: application/json; charset=utf-8');
			header('HTTP/1.1 403 Forbidden');
			http_response_code($validated_client['code'] ?? 403);

			echo
			json_encode([
				'success'    => FALSE,
				'error'      => $validated_client['error'],
				'error_code' => $validated_client['error_code'],
				'code'       => 403,
			]);
			exit();
		}

		$valid_ip = $this->validate_request_ip($validated_client);
		if ($valid_ip !== TRUE)
		{
			header('Content-Type: application/json; charset=utf-8');
			header('HTTP/1.1 403 Forbidden');
			http_response_code($validated_client['code'] ?? 403);

			echo
			json_encode(array_merge([
				'success' => FALSE,
			], $valid_ip));

			exit();
		}

		$landlord       = NULL;
		$landlord_email = $this->input->get('landlord_email', TRUE);
		if ($landlord_email)
		{
			$landlord = $this->get_landlord($landlord_email);
			if ( ! $landlord)
			{
				header('Content-Type: application/json; charset=utf-8');
				header('HTTP/1.1 404 Not Found');
				http_response_code(404);

				echo
				json_encode([
					'success' => FALSE,
					'msg'     => 'Landlord not found',
				]);

				exit();
			}
		}

		$order_by        = $this->input->get('order');
		$order_direction = $this->input->get('direction');
		$properties      = $this->get_properties($landlord, $order_by, $order_direction);
		header('Content-Type: application/json; charset=utf-8');
		header('HTTP/1.1 200 Ok');
		http_response_code(200);
		echo
		json_encode([
			'success' => TRUE,
			'data'    => [
				'landlord'   => $landlord,
				'properties' => $properties,
			],
		]);
		exit();
	}

	private function get_properties($landlord = NULL, $order = 'id', $direction = 'asc')
	{
		$CI             = &get_instance();
		$valid_order_by = ['id', 'property_name', 'created_at', 'updated_at', 'property_address', 'property_type_id'];
		switch ($order)
		{
			case 'id':
				$order = 'propertyid';
				break;
			case 'name':
				$order = 'property_name';
				break;
			case 'created_date':
				$order = 'created_at';
				break;
			case 'updated_date':
				$order = 'updated_at';
				break;
			case 'property_address':
				$order = 'property_address';
				break;
			case 'property_type':
				$order = 'property_type_id';
				break;
			default:
				break;
		}
		if ( ! in_array($order, $valid_order_by))
		{
			$order = 'id';
		}

		$direction = $direction == 'asc' ? $direction : 'desc';

		$properties_table_name = db_prefix().'tjm_properties';

		$CI->db->select(
			"propertyid as id,property_name, description as notes,landlord_id,{$properties_table_name}.created_at as created_date, {$properties_table_name}.updated_at as updated_date, property_address, type_name as property_type",
		);
		$CI->db->join(db_prefix().'tjm_property_types', db_prefix().'tjm_property_types.propertytypeid = '.db_prefix().'tjm_properties.property_type_id');
		$CI->db->order_by($order, $direction);

		if ($landlord)
		{
			$CI->db->where('landlord_id', $landlord['landlordid']);
		}

		return $CI->db->get(db_prefix().'tjm_properties')->result_array();
	}

	private function get_property($landlord, $property_name)
	{
		$CI = &get_instance();

		$properties_table_name = db_prefix().'tjm_properties';

		$CI->db->select(
			"propertyid as id,property_name, description as notes,landlord_id,{$properties_table_name}.created_at as created_date, {$properties_table_name}.updated_at as updated_date, property_address, type_name as property_type",
		);
		$CI->db->select("propertyid as id,property_name as name");
		$CI->db->where('landlord_id', $landlord['landlordid']);
		$CI->db->join(db_prefix().'tjm_property_types', db_prefix().'tjm_property_types.propertytypeid = '.$properties_table_name.'.property_type_id');
		$CI->db->like('property_name', $property_name);

		return $CI->db->get($properties_table_name)->row_array();
	}

	/**
	 * @param $landlord
	 * @param $property_name
	 *
	 * @return array|false
	 */
	private function create_property($landlord, $property_name)
	{
		$property_type = $this->db->get(db_prefix().'tjm_property_types')->row_array();
		$data          = [
			'property_name'    => $property_name,
			'description'      => $property_name,
			'property_type_id' => $property_type['propertytypeid'],
			'landlord_id'      => $landlord['landlordid'],
			'property_address' => sprintf('%s %s %s %s', $landlord['address'], $landlord['city'], $landlord['zip'], $landlord['state']),
		];

		if ($landlord['country'])
		{
			$country = $landlord['country'];

			$data['property_address'] .= ' '.get_country_name($country);
		}

		$id = $this->properties_model->add($data);
		if ($id)
		{
			return (array)$this->properties_model->get($id);
		}

		return FALSE;
	}

	private function validate_request_client()
	{
		$authorization = $this->input->get_request_header('Authorization');
		if (empty($authorization))
		{
			$authorization = $this->input->get_request_header('X-Auth-Token');
		}
		if ( ! str_starts_with($authorization, 'Basic '))
		{
			return [
				'error'      => 'Permission denied',
				'error_code' => 'authorization_header',
				'code'       => 403,
			];
		}

		$api_clients = tjm_get_all_api_clients();

		if (empty($api_clients))
		{
			return [
				'error'      => 'Permission denied',
				'error_code' => 'empty_clients',
				'code'       => 403,
			];
		}

		$authorization = str_replace('Basic ', '', $authorization);

		foreach ($api_clients as $api_client)
		{
			$basic_auth = base64_encode($api_client['client_id'].':'.$api_client['client_secret']);

			if ($basic_auth == $authorization)
			{
				return $api_client;
			}
		}

		return [
			'error'      => 'Permission denied',
			'error_code' => 'validate_request_client',
			'code'       => 403,
		];
	}

	/**
	 * @param  array  $client
	 *
	 * @return array|true
	 */
	private function validate_request_ip(array $client): bool|array
	{
		$client_ip = $client['client_ip'];
		if (empty($client_ip) || $client_ip === '*')
		{
			return TRUE;
		}

		$request_ip = $this->input->ip_address();
		if ($request_ip === '127.0.0.1')
		{
			return TRUE;
		}

		$client_ips = explode(',', $client_ip);

		if ( ! is_array($client_ips) && $request_ip == $client_ips || is_array($client_ips) && in_array($request_ip, $client_ips))
		{
			return TRUE;
		}

		return [
			'error'      => 'IP address is not allowed',
			'request_ip' => $request_ip,
			'code'       => 403,
		];
	}

	/**
	 * Validate request data
	 *
	 * @param  array  $data
	 *
	 * @return array
	 */
	private function validate_request_data(array $data): array
	{
		$valid_landlord = $this->validate_landlord_data($data);
		if (isset($valid_landlord['error']))
		{
			return $valid_landlord;
		}

		$valid_job = $this->validate_job_data($data);
		if (isset($valid_job['error']))
		{
			return $valid_job;
		}

		return [
			'success'  => TRUE,
			'landlord' => $valid_landlord,
			'job'      => $valid_job,
		];
	}

	/**
	 * Validate valid landlord
	 *
	 * @param  array  $data
	 *
	 * @return array
	 */
	private function validate_landlord_data(array $data): array
	{
		$required_fields = ['firstname', 'lastname', 'contact_email', 'phonenumber'];
		$optional_fields = ['address', 'city', 'state', 'zip', 'country', 'website'];

		foreach ($required_fields as $field)
		{
			if (empty($data[$field]))
			{
				return [
					'error' => $field.' is required',
					'code'  => 400,
				];
			}
		}

		$email = $data['contact_email'];
		if ( ! filter_var($email, FILTER_VALIDATE_EMAIL))
		{
			return [
				'error' => 'Invalid email address',
				'code'  => 400,
			];
		}

		$return_data = [];
		foreach ($required_fields as $field)
		{
			$return_data[$field] = $data[$field];
		}

		$country = $data['country'] ?? FALSE;
		if ($country !== FALSE)
		{
			$countries = get_all_countries();
			$countries = array_map(function ($country) {
				return $country['short_name'];
			}, $countries);

			if ( ! in_array($country, $countries))
			{
				return [
					'error'     => 'Invalid country name',
					'countries' => $countries,
					'code'      => 400,
				];
			}
		}
		foreach ($optional_fields as $field)
		{
			$return_data[$field] = $data[$field] ?? NULL;
		}

		return $return_data;
	}

	/**
	 * Validate valid job
	 *
	 * @param  array  $data
	 *
	 * @return array
	 */
	private function validate_job_data(array $data)
	{
		$required_fields = [
			'property_name',
			'job_type',
			'job_priority',
			'job_name',
			'visit_date',
			'contact_name',
			'contact_phone_number',
		];

		foreach ($required_fields as $field)
		{
			if (empty($data[$field]))
			{
				return [
					'error' => $field.' is required',
					'code'  => 400,
				];
			}
		}

		$job_type = $data['job_type'];
		if (is_array($job_type))
		{
			$job_type         = array_shift($job_type);
			$data['job_type'] = $job_type;
		}
		$job_type = $this->job_types_model->get_by_name($job_type, TRUE);
		if ( ! $job_type)
		{
			return [
				'error'     => 'Invalid job type',
				'job_types' => $this->job_types_model->get('', TRUE),
				'code'      => 400,
			];
		}

		if (is_array($data['job_priority']))
		{
			$data['job_priority'] = array_shift($data['job_priority']);
		}

		$data['job_priority'] = match ($data['job_priority'])
		{
			'Low' => 1,
			'Medium' => 2,
			'High' => 3,
			'Urgent' => 4,
			default => $data['job_priority'],
		};

		$job_priority = $data['job_priority'];

		$all_job_priorities = [1, 2, 3, 4];

		if ( ! in_array($job_priority, $all_job_priorities))
		{
			return [
				'error'          => 'Invalid job priority',
				'job_priorities' => 'Priorities: '.implode(',', $all_job_priorities),
				'code'           => 400,
			];
		}

		$visit_date = $data['visit_date'];
		$visit_date = strtotime($visit_date);
		if ( ! $visit_date)
		{
			return [
				'error' => 'Invalid preferred date to visit',
				'code'  => 400,
			];
		}

		$return_data = [];
		foreach ($required_fields as $field)
		{
			$return_data[$field] = trim($data[$field]);
		}

		return $return_data;
	}

}
