<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Trade Job Management - Job Confirmation Helper
 *
 * Provides functions for job confirmation, digital signature handling,
 * and client satisfaction tracking
 */

/**
 * Get acceptance information array for saving to database
 * Similar to estimate acceptance
 *
 * @return array
 */
function get_job_acceptance_info_array()
{
	$CI = &get_instance();

	return [
		'signature_path'       => $GLOBALS['processed_digital_signature'],
		'confirmation_firstname' => $CI->input->post('confirmation_firstname'),
		'confirmation_lastname'  => $CI->input->post('confirmation_lastname', ''),
		'confirmation_email'     => $CI->input->post('confirmation_email'),
		'confirmation_feedback'  => $CI->input->post('confirmation_feedback', ''),
		'confirmation_signed'    => 1,
		'confirmation_at'      => date('Y-m-d H:i:s'),
	];
}

/**
 * Process digital signature image and save to file
 * Saves the signature as an image file in the job uploads directory
 *
 * @param  string  $signature_data  Base64 encoded signature data
 * @param  int  $job_id  Job ID
 *
 * @return bool
 */
function process_job_digital_signature($signature_data, $job_id)
{
	$CI = &get_instance();

	if (empty($signature_data))
	{
		return FALSE;
	}

	// Create uploads directory if not exists
	$upload_path = FCPATH.'uploads/jobs/'.$job_id;
	if ( ! is_dir($upload_path))
	{
		mkdir($upload_path, 0755, TRUE);
	}

	// Process the signature image
	return process_digital_signature_image($signature_data, $upload_path);
}

/**
 * Check if job can be confirmed by client
 * Job should not be completed or exceed NTE
 *
 * @param  int  $job_id  Job ID
 *
 * @return bool
 */
function can_job_be_confirmed($job_id)
{
	$CI = &get_instance();
	$CI->load->model('trade_job_management/jobs_model');

	$job = $CI->jobs_model->get($job_id);

	if ( ! $job)
	{
		return FALSE;
	}

	// Check if job is in a state that allows confirmation
	if ($job->status == Jobs_model::STATUS_COMPLETE || $job->status == Jobs_model::STATUS_EXCEED_NTE)
	{
		return FALSE;
	}

	return TRUE;
}

/**
 * Get job confirmation details
 *
 * @param  int  $job_id  Job ID
 *
 * @return object|null
 */
function get_job_confirmation_details($job_id)
{
	$CI = &get_instance();
	$CI->load->model('trade_job_management/jobs_model');

	return $CI->jobs_model->get($job_id);
}

/**
 * Format job confirmation status
 *
 * @param  bool  $is_confirmed  Whether job is confirmed
 *
 * @return string
 */
function format_job_confirmation_status($is_confirmed)
{
	if ($is_confirmed)
	{
		return '<span class="badge badge-success"><i class="fa fa-check-circle"></i> '._l('confirmed').'</span>';
	}

	return '<span class="badge badge-default"><i class="fa fa-circle-o"></i> '._l('not_confirmed').'</span>';
}

/**
 * Send job confirmation email to client
 * Notifies client when job has been marked as confirmed
 *
 * @param  int  $job_id  Job ID
 *
 * @return bool
 */
function send_job_confirmation_email($job_id)
{
	$CI = &get_instance();
	$CI->load->model('trade_job_management/jobs_model');

	$job = $CI->jobs_model->get($job_id);

	if ( ! $job)
	{
		return FALSE;
	}

	// You can customize this email sending logic
	// For now, just log the confirmation
	log_activity('Job Confirmed by Client [ID: '.$job_id.'] - '.$CI->input->post('confirmation_firstname').' '.$CI->input->post('confirmation_lastname'));

	return TRUE;
}

/**
 * Get job confirmation history
 * Returns all confirmation records for a job
 *
 * @param  int  $job_id  Job ID
 *
 * @return array
 */
function get_job_confirmation_history($job_id)
{
	$CI = &get_instance();

	$CI->db->where('jobid', $job_id);
	$history = $CI->db->get(db_prefix().'tjm_job_confirmations')->result_array();

	return $history;
}

/**
 * Check if job signature file exists
 *
 * @param  int  $job_id  Job ID
 *
 * @return bool
 */
function job_signature_exists($job_id)
{
	$signature_path = FCPATH.'uploads/jobs/'.$job_id;
	$files          = glob($signature_path.'/signature_*');

	return ! empty($files);
}

/**
 * Get job signature file path
 *
 * @param  int  $job_id  Job ID
 *
 * @return string|null
 */
function get_job_signature_path($job_id)
{
	$signature_path = FCPATH.'uploads/jobs/'.$job_id;
	$files          = glob($signature_path.'/signature_*');

	if ( ! empty($files))
	{
		return $files[0];
	}

	return NULL;
}

/**
 * Verify client access to job for confirmation
 * Checks hash and permissions
 *
 * @param  int  $job_id  Job ID
 * @param  string  $hash  Access hash
 *
 * @return bool
 */
function verify_client_job_access($job_id, $hash = '')
{
	$CI = &get_instance();

	// Check if client is logged in or has valid hash access
	if (is_client_logged_in())
	{
		return TRUE;
	}

	// If hash provided, verify it
	if ( ! empty($hash))
	{
		// Add your hash verification logic here
		return true;
	}

	return false;
}
