<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @property Jobs_model $jobs_model
 * @property Job_daily_logs_model $job_daily_logs_model
 * @property Misc_model misc_model
 */
class Jobs extends AdminController {

	protected const BASE_PATH = '/jobs';

	public function __construct()
	{
		parent::__construct();

		$this->load->model('landlords_model');
		$this->load->model('jobs_model');
		$this->load->model('job_types_model');
		$this->load->model('properties_model');
		$this->load->model('job_status_logs_model');
		$this->load->model('job_scheduled_logs_model');
		$this->load->model('invoices_model');
		$this->load->model('job_timers_model');
		$this->load->model('job_daily_logs_model');
		$this->load->model('misc_model');
		$this->load->helper('date_helper');
	}

	/* Open also all jobs if user access this /tasks url */

	public function index()
	{
		if (staff_cant('view', 'tasks'))
		{
			access_denied('jobs');
		}
		$this->list_jobs();
	}

	public function list_jobs($id = '')
	{
		close_setup_menu();
		$data['jobid'] = $id;

		if ($this->input->get('kanban'))
		{
			$this->switch_kanban(0, TRUE);
		}

		$data['bodyclass']  = 'jobs-page';
		$data['title']      = _l('jobs');
		$data['jobs_table'] = App_table::find('jobs');
		$this->load->view(static::BASE_PATH.'/index', $data);
	}

	/* List all jobs */

	public function show($id)
	{
		if (staff_cant('view', 'tasks'))
		{
			access_denied('jobs');
		}
		$this->list_jobs($id);
	}

	public function table()
	{
		App_table::find('jobs')->output();
	}

	public function create($id = '')
	{
		if (staff_cant('create', 'tasks'))
		{
			ajax_access_denied();
		}

		$data = [];

		if ($this->input->post(NULL, FALSE))
		{
			hooks()->add_filter('html_purify_content', function ($will_purify) {
				return FALSE;
			});
			$data                    = $this->input->post(NULL, FALSE);
			$data['job_description'] = html_purify($this->input->post('job_description', FALSE));
			if (staff_cant('create', 'tasks'))
			{
				header('HTTP/1.0 400 Bad error');
				echo
				json_encode([
					'success' => FALSE,
					'message' => _l('access_denied'),
				]);
				die();
			}

			if ( ! ($data['property_id'] ?? FALSE))
			{
				echo
				json_encode([
					'success' => FALSE,
					'message' => _l('job_select_property'),
				]);
				die();
			}

			unset($data['rel_type']);

			$id = $this->jobs_model->add($data);

			$_id     = FALSE;
			$success = FALSE;
			$message = '';
			if ($id)
			{
				$job = $this->jobs_model->get($id);
				send_job_reported_email($job);

				$success       = TRUE;
				$_id           = $id;
				$message       = _l('added_successfully', _l('job'));
				$uploadedFiles = handle_task_attachments_array($id);
				if ($uploadedFiles && is_array($uploadedFiles))
				{
					foreach ($uploadedFiles as $file)
					{
						$this->misc_model->add_attachment_to_database($id, 'task', [$file]);
					}
				}
			}
			echo
			json_encode([
				'success' => $success,
				'id'      => $_id,
				'message' => $message,
			]);
			die();
		}

		$data['members']        = $this->staff_model->get();
		$data['id']             = $id;
		$data['title']          = _l('add_new', _l('job_lowercase'));
		$data['job_types']      = $this->job_types_model->get('', TRUE);
		$data['job_priorities'] = get_tasks_priorities();

		$this->load->view('jobs/create', $data);
	}

	public function edit($id = '')
	{
		if (staff_cant('edit', 'tasks'))
		{
			ajax_access_denied();
		}

		if ($this->input->post(NULL, FALSE))
		{
			hooks()->add_filter('html_purify_content', function ($will_purify) {
				return FALSE;
			});
			$data                    = $this->input->post(NULL, FALSE);
			$data['job_description'] = html_purify($this->input->post('job_description', FALSE));
			if (staff_cant('edit', 'tasks'))
			{
				header('HTTP/1.0 400 Bad error');
				echo
				json_encode([
					'success' => FALSE,
					'message' => _l('access_denied'),
				]);
				die();
			}

			$success = $this->jobs_model->update($data, $id);
			$message = '';
			if ($success)
			{
				$message = _l('updated_successfully', _l('job'));
			}
			echo
			json_encode([
				'success' => $success,
				'message' => $message,
				'id'      => $id,
			]);
			die();
		}

		$job         = $this->jobs_model->get($id);
		$data['job'] = $job;
		$title       = _l('edit', _l('job_lowercase')).' '.$data['task']->name;

		$data['id']             = $id;
		$data['members']        = $this->staff_model->get();
		$data['title']          = $title;
		$data['specialist_ids'] = $job->specialist_ids;

		$property               = $this->properties_model->get($data['job']->property_id);
		$data['property']       = $property;
		$data['job_types']      = $this->job_types_model->get('', TRUE);
		$data['job_priorities'] = get_tasks_priorities();

		$this->load->view('jobs/edit', $data);
	}

	public function change_priority($priority_id, $id)
	{
		if (staff_can('edit', 'tasks'))
		{
			$this->db->where('jobid', $id);
			$this->db->update(db_prefix().'tjm_jobs', ['priority' => $priority_id]);

			$success = $this->db->affected_rows() > 0 ? TRUE : FALSE;

			// Don't do this query if the action is not performed via task single
			$jobHtml = $this->input->get('single_job') === 'true' ? $this->get_job_data($id, TRUE) : '';
			echo
			json_encode([
				'success' => $success,
				'jobHtml' => $jobHtml,
			]);
		} else
		{
			echo
			json_encode([
				'success' => FALSE,
				'jobHtml' => '',
			]);
		}
	}

	/**
	 * Job ajax request modal
	 *
	 * @param  mixed  $jobid
	 *
	 * @return mixed
	 */
	public function get_job_data($jobid, $return = FALSE)
	{
		$jobs_where = [];

		if (staff_cant('view', 'tasks'))
		{
			$jobs_where = get_jobs_where_string(FALSE);
		}

		$job = $this->jobs_model->get($jobid, $jobs_where);

		if ( ! $job)
		{
			header('HTTP/1.0 404 Not Found');
			echo 'Task not found';
			die();
		}

		$data['job']      = $job;
		$data['id']       = $job->jobid;
		$data['staff']    = $this->staff_model->get('', ['active' => 1]);
		$data['property'] = $this->properties_model->get($job->property_id);
		$data['landlord'] = $this->landlords_model->get($job->property['landlord_id']);

		$data['statuses'] = $this->jobs_model->get_statuses();

		$status_logs = $this->job_status_logs_model->get_by_job_id($job->jobid);
		$status_logs = array_map(function ($log) {
			$log->type = 'status';

			return $log;
		}, $status_logs);

		$scheduled_logs = $this->job_scheduled_logs_model->get_by_job_id($job->jobid);
		$scheduled_logs = array_map(function ($log) {
			$log->type = 'schedule';

			return $log;
		}, $scheduled_logs);

		$activity_logs = array_merge($status_logs, $scheduled_logs);

		usort($activity_logs, function ($a, $b) {
			if ($a->created_at === $b->created_at)
			{
				return 0;
			}

			return strcmp($b->created_at, $a->created_at);
		});

		$data['activity_logs'] = $activity_logs;

		if ($return == FALSE)
		{
			$this->load->view('jobs/view_job_template', $data);
		} else
		{
			return $this->load->view('jobs/view_job_template', $data, TRUE);
		}
	}

	/* Add new job comment / ajax */

	/**
	 * Job ajax request modal
	 *
	 * @param  mixed  $jobid
	 *
	 * @return mixed
	 */
	public function invoice($job_id)
	{
		$jobs_where = [];

		if (staff_cant('view', 'tasks'))
		{
			$jobs_where = get_jobs_where_string(FALSE);
		}

		$job = $this->jobs_model->get($job_id, $jobs_where);

		if ( ! $job)
		{
			header('HTTP/1.0 404 Not Found');
			echo 'Job not found';
			die();
		}

		$title = _l('create_new_invoice').' for job:  '.$job->job_title;

		$data['job']       = $job;
		$data['countries'] = $this->clients_model->get_clients_distinct_countries();
		$this->load->model('payment_modes_model');
		$data['payment_modes'] = $this->payment_modes_model->get('', [
			'expenses_only !=' => 1,
		]);

		$this->load->model('taxes_model');
		$data['taxes'] = $this->taxes_model->get();
		$this->load->model('invoice_items_model');
		$this->load->model('currencies_model');
		$data['currencies']    = $this->currencies_model->get();
		$data['base_currency'] = $this->currencies_model->get_base_currency();
		$data['staff']         = $this->staff_model->get('', ['active' => 1]);
		$data['title']         = $title;
		$data['bodyclass']     = 'invoice';

		$this->load->view('jobs/invoice', $data);
	}

	public function save_invoice($job_id = '')
	{
		if ($this->input->post())
		{
			$invoice_data = $this->input->post();
			if (staff_cant('create', 'invoices'))
			{
				access_denied('invoices');
			}

			if (hooks()->apply_filters('validate_invoice_number', TRUE))
			{
				$number = ltrim($invoice_data['number'], '0');
				if (total_rows('invoices', [
					'YEAR(date)' => (int)date('Y', strtotime(to_sql_date($invoice_data['date']))),
					'number'     => $number,
					'status !='  => Invoices_model::STATUS_DRAFT,
				])
				)
				{
					set_alert('warning', _l('invoice_number_exists'));

					redirect(admin_url('trade_job_management/jobs/invoice/'.$job_id));
				}
			}

			$id = $this->invoices_model->add($invoice_data);
			if ($id)
			{
				$this->jobs_model->update_invoice($job_id, $id);

				set_alert('success', _l('added_successfully', _l('invoice')));
				$redirect_to = admin_url('invoices/list_invoices/'.$id);

				if (isset($invoice_data['save_and_record_payment']))
				{
					$this->session->set_userdata('record_payment', TRUE);
				} elseif (isset($invoice_data['save_and_send_later']))
				{
					$this->session->set_userdata('send_later', TRUE);
				}

				redirect($redirect_to);
				die();
			}
		}

		redirect(admin_url('trade_job_management/jobs/show/'.$job_id));
		die();
	}

	public function add_job_comment()
	{
		$data = $this->input->post();

		hooks()->add_filter('html_purify_content', function ($will_purify) {
			return FALSE;
		});

		$data['content'] = html_purify($this->input->post('content', FALSE));
		if ($this->input->post('no_editor'))
		{
			$data['content'] = nl2br($this->input->post('content'));
		}
		$comment_id = FALSE;
		if ($data['content'] != '' || isset($_FILES['file']['name']) && is_array($_FILES['file']['name']) && count($_FILES['file']['name']) > 0)
		{
			$comment_id = $this->jobs_model->add_job_comment($data);
			if ($comment_id)
			{
				$comment_attachments = handle_job_attachments_array($data['jobid'], 'file');
				if ($comment_attachments && is_array($comment_attachments))
				{
					foreach ($comment_attachments as $file)
					{
						$attachment_id = $this->misc_model->add_attachment_to_database($data['jobid'], 'job', [$file]);
						$this->jobs_model->update_attachment_job_comment_id($attachment_id, $comment_id);
					}

					if (count($comment_attachments) > 0)
					{
						$this->db->query(
							'UPDATE '.db_prefix()."tjm_job_comments SET content = CONCAT(content, '[job_attachment]')
                            WHERE jcid = ".$this->db->escape_str($comment_id),
						);
					}
				}
			}
		}
		echo
		json_encode([
			'success' => $comment_id ? TRUE : FALSE,
			'jobHtml' => $this->get_job_data($data['jobid'], TRUE),
		]);
	}

	public function delete($id)
	{
		if (staff_cant('delete', 'tasks'))
		{
			access_denied('tasks');
		}
		$success = $this->jobs_model->delete_job($id);
		$message = _l('problem_deleting', _l('job_lowercase'));
		if ($success)
		{
			$message = _l('deleted', _l('job'));
			set_alert('success', $message);
		} else
		{
			set_alert('warning', $message);
		}

		if (
			empty($_SERVER['HTTP_REFERER'])
			|| strpos($_SERVER['HTTP_REFERER'], 'trade_job_management/jobs') !== FALSE
			|| strpos($_SERVER['HTTP_REFERER'], 'trade_job_management/jobs/index') !== FALSE
			|| strpos($_SERVER['HTTP_REFERER'], 'trade_job_management/jobs/show') !== FALSE
		)
		{
			redirect(admin_url('trade_job_management'.static::BASE_PATH));
		} else
		{
			redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
		}
	}

	/* Delete task from database */

	public function job_reschedule($id)
	{
		if (staff_can('edit', 'tasks'))
		{
			$job                = $this->jobs_model->get($id);
			$old_scheduled_time = $job->preferred_visit_date;

			$post_data                    = $this->input->post();
			$new_time                     = to_sql_date($post_data['preferred_visit_date'], TRUE);
			$data['preferred_visit_date'] = $new_time;

			$this->db->where('jobid', $id);
			$this->db->update(db_prefix().'tjm_jobs', $data);

			$this->job_scheduled_logs_model->add([
				'job_id'             => $id,
				'old_scheduled_time' => $old_scheduled_time,
				'new_scheduled_time' => $new_time,
			]);

			if ( ! $old_scheduled_time || $job->status == Jobs_model::STATUS_REPORTED)
			{
				$this->jobs_model->mark_as(Jobs_model::STATUS_SCHEDULED, $id);
			}

			$job = $this->jobs_model->get($id);
			send_scheduled_time_email($job);

			$formatted = Datetime::createFromFormat('Y-m-d H:i:s', $job->preferred_visit_date);
			$formatted = $formatted->format('d F Y, h:ia');
			echo
			json_encode([
				'success'      => TRUE,
				'new_schedule' => $formatted,
				'message'      => _l('job_rescheduled'),
			]);
			die();
		}
	}

	public function mark_as($status, $id)
	{
		if ($this->jobs_model->is_job_specialist(get_staff_user_id(), $id)
		    || $this->jobs_model->is_job_creator(get_staff_user_id(), $id)
		    || staff_can('edit', 'tasks')
		)
		{
			$job        = $this->jobs_model->get($id);
			$old_status = $job->status;

			$success = $this->jobs_model->mark_as($status, $id);

			// Don't do this query if the action is not performed via task single
			$jobHtml = $this->input->get('single_job') === 'true' ? $this->get_job_data($id, TRUE) : '';

			if ($success)
			{
				$message = _l('job_marked_as_success', format_job_status($status, TRUE, TRUE));
				$this->job_status_logs_model->add([
					'job_id'     => $id,
					'old_status' => $old_status,
					'new_status' => $status,
				]);

				if ($status == Jobs_model::STATUS_COMPLETE)
				{
					$job = $this->jobs_model->get($id);
					send_job_completed_email($job);
				} elseif ($status == Jobs_model::STATUS_REPORTED)
				{
					$job = $this->jobs_model->get($id);
					send_job_reported_email($job);
				}
				echo json_encode([
					'success' => TRUE,
					'message' => $message,
					'jobHtml' => $jobHtml,
				]);
				exit();
			} else
			{
				$CI      = &get_instance();
				$message = $CI->session->userdata('danger-alert');
				echo json_encode([
					'success' => FALSE,
					'message' => $message,
					'jobHtml' => $jobHtml,
				]);
			}
		} else
		{
			echo
			json_encode([
				'success' => FALSE,
				'message' => '',
				'jobHtml' => '',
			]);
		}
	}

	/**
	 * Upload task attachment
	 *
	 * @since  Version 1.0.1
	 */
	public function upload_file()
	{
		if ($this->input->post())
		{
			$jobid   = $this->input->post('jobid');
			$files   = handle_job_attachments_array($jobid, 'file');
			$success = FALSE;

			if ($files)
			{
				$i   = 0;
				$len = count($files);
				foreach ($files as $file)
				{
					$success = $this->jobs_model->add_attachment_to_database($jobid, [$file], FALSE, $i == ($len - 1) ? TRUE : FALSE);
					$i++;
				}
			}

			echo
			json_encode([
				'success' => $success,
				'jobHtml' => $this->get_job_data($jobid, TRUE),
			]);
		}
	}

	public function edit_comment()
	{
		if ($this->input->post())
		{
			$data = $this->input->post();
			hooks()->add_filter('html_purify_content', function ($will_purify) {
				return FALSE;
			});
			$data['content'] = html_purify($this->input->post('content', FALSE));
			if ($this->input->post('no_editor'))
			{
				$data['content'] = nl2br(clear_textarea_breaks($this->input->post('content')));
			}
			$comment_id = $data['id'] ?? '';
			$job_id     = $data['job_id'] ?? '';
			if (empty($comment_id) || empty($job_id))
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('job_not_found'),
					'jobHtml' => $this->get_job_data($data['job_id'], TRUE),
				]);
				exit();
			}
			$success = $this->jobs_model->edit_job_comment($data);
			$message = '';
			if ($success)
			{
				$message = _l('task_comment_updated');
			}
			echo json_encode([
				'success' => $success,
				'message' => $message,
				'jobHtml' => $this->get_job_data($data['job_id'], TRUE),
			]);
		}
	}

	public function get_billable_job_data($job_id)
	{
		$job = $this->jobs_model->get_billable_job_data($job_id);

		echo json_encode($job);
	}

	public function update_job_description($id)
	{
		if (staff_can('edit', 'tasks'))
		{
			$data = hooks()->apply_filters(
				'before_update_job',
				[
					'job_description' => html_purify($this->input->post('job_description', FALSE)),
				],
				$id,
			);

			$this->db->where('jobid', $id);
			$this->db->update(db_prefix().'tjm_jobs', $data);

			hooks()->do_action('after_update_job', $id);
		}
	}

	public function timer_tracking()
	{
		$job_id     = $this->input->post('job_id');
		$admin_stop = $this->input->get('admin_stop') && is_admin() ? TRUE : FALSE;

		if ($admin_stop)
		{
			$this->session->set_flashdata('job_single_timesheets_open', TRUE);
		}

		$result = $this->job_timers_model->timer_tracking(
			$job_id,
			$this->input->post('timer_id'),
			nl2br($this->input->post('note')),
			floatval($this->input->post('cost')),
			$admin_stop,
		);

		if ($result)
		{
			$this->update_job_exceed_nte_status($job_id);
		}

		echo json_encode([
			'success'  => $result,
			'taskHtml' => $this->get_job_data($job_id, TRUE),
			'timers'   => $this->get_staff_started_timers(TRUE),
		]);
	}

	public function delete_user_unfinished_timesheet($id)
	{
		$this->db->where('timerid', $id);
		$timesheet = $this->db->get(db_prefix().'tjm_job_timers')->row();
		if ($timesheet && $timesheet->end_time == NULL && $timesheet->staff_id == get_staff_user_id())
		{
			$this->db->where('timerid', $id);
			$this->db->delete(db_prefix().'tjm_job_timers');
		}
		echo json_encode(['timers' => $this->get_staff_started_timers(TRUE)]);
	}

	public function delete_timesheet($id)
	{
		if (staff_can('delete_timesheet', 'tasks') || staff_can('delete_own_timesheet', 'tasks') && total_rows(db_prefix().'tjm_job_timers', ['staff_id' => get_staff_user_id(), 'timerid' => $id]) > 0)
		{
			$alert_type = 'warning';
			$success    = $this->job_timers_model->delete_timesheet($id);
			if ($success)
			{
				$this->session->set_flashdata('job_single_timesheets_open', TRUE);
				$message = _l('deleted', _l('job_timesheet'));
				set_alert('success', $message);
			}
			if ( ! $this->input->is_ajax_request())
			{
				redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
			}
		}
	}

	public function update_timesheet()
	{
		if ($this->input->is_ajax_request())
		{
			if (staff_can('edit_timesheet', 'tasks')
			    || (staff_can('edit_own_timesheet', 'tasks') && total_rows(db_prefix().'tjm_job_timers', ['staff_id' => get_staff_user_id(), 'timerid' => $this->input->post('timer_id')]) > 0)
			)
			{
				$success = $this->job_timers_model->timesheet($this->input->post());
				if ($success === TRUE)
				{
					$job_id = $this->input->post('job_id');
					$this->update_job_exceed_nte_status($job_id);

					$this->session->set_flashdata('job_single_timesheets_open', TRUE);
					$message = _l('updated_successfully', _l('job_timesheet'));
				} else
				{
					$message = _l('failed_to_update_timesheet');
				}

				echo json_encode([
					'success' => $success,
					'message' => $message,
				]);
				die;
			}

			echo json_encode([
				'success' => FALSE,
				'message' => _l('access_denied'),
			]);
			die;
		}
	}

	public function log_time()
	{
		$success = $this->job_timers_model->timesheet($this->input->post());
		if ($success === TRUE)
		{
			$job_id = $this->input->post('job_id');
			$this->update_job_exceed_nte_status($job_id);

			$this->session->set_flashdata('job_single_timesheets_open', TRUE);
			$message = _l('added_successfully', _l('job_timesheet'));
		} elseif (is_array($success) && isset($success['end_time_smaller']))
		{
			$message = _l('failed_to_add_project_timesheet_end_time_smaller');
		} else
		{
			$message = _l('job_timesheet_not_updated');
		}

		echo json_encode([
			'success' => $success,
			'message' => $message,
		]);
		die;
	}

	public function get_staff_started_timers($return = FALSE)
	{
		$data['started_timers'] = $this->job_timers_model->get_staff_started_timers();
		$_data['html']          = $this->load->view('jobs/started_timers', $data, TRUE);
		$_data['total_timers']  = count($data['started_timers']);

		$timers = json_encode($_data);
		if ($return)
		{
			return $timers;
		}

		echo $timers;
	}

	private function update_job_exceed_nte_status($job_id)
	{
		$nte        = $this->jobs_model->get_nte($job_id);
		$timesheets = $this->jobs_model->get_timesheets($job_id);
		$total_cost = array_sum(array_column($timesheets, 'total'));

		if (floatval($total_cost) > $nte)
		{
			$this->jobs_model->mark_as(Jobs_model::STATUS_EXCEED_NTE, $job_id);
		}
	}

	/**
	 * Daily logs main page for a job
	 *
	 * @param  int  $job_id  job id
	 */
	public function daily_logs($job_id)
	{
		if (staff_cant('view', 'tasks'))
		{
			access_denied('jobs');
		}

		$job = $this->jobs_model->get($job_id);

		if ( ! $job)
		{
			show_404();
		}

		$data['job']        = $job;
		$data['daily_logs'] = $this->job_daily_logs_model->get_by_job_id($job_id);
		$data['statistics'] = $this->job_daily_logs_model->get_statistics($job_id);
		$data['title']      = _l('daily_progress').' - '.$job->job_title;

		$this->load->view('jobs/daily_logs/index', $data);
	}

	/**
	 * Submit daily log form
	 *
	 * @param  int  $job_id  job id
	 */
	public function submit_daily_log($job_id = '')
	{
		// Show form
		if (staff_cant('view', 'tasks'))
		{
			access_denied('jobs');
		}

		$job = $this->jobs_model->get($job_id);

		if ( ! $job)
		{
			show_404();
		}

		$data['job']   = $job;
		$data['title'] = _l('submit_daily_log').' - '.$job->job_title;

		$this->load->view('jobs/daily_logs/submit', $data);
	}

	public function handle_ajax_submit_daily_log($job_id = '')
	{
		if ($this->input->post())
		{
			$data = $this->input->post();

			// Check permissions
			if ( ! $this->jobs_model->is_job_specialist(get_staff_user_id(), $data['job_id'])
			     && ! $this->jobs_model->is_job_creator(get_staff_user_id(), $data['job_id'])
			     && ! staff_can('create', 'tasks')
			)
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('access_denied'),
				]);
				die();
			}

			// Validate required fields
			if (empty($data['comments']))
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_comments_required'),
				]);
				die();
			}

			// HTML purify comments
			hooks()->add_filter('html_purify_content', function ($will_purify) {
				return FALSE;
			});
			$data['comments'] = html_purify($this->input->post('comments', FALSE));

			// Convert date format
			if ( ! empty($data['log_date']))
			{
				$data['log_date'] = to_sql_date($data['log_date']);
			}

			$daily_log_id = $this->job_daily_logs_model->add($data);

			if ($daily_log_id)
			{
				// Handle photo uploads
				$uploaded_files = handle_job_daily_log_attachments($data['job_id'], $daily_log_id);
				if ($uploaded_files && is_array($uploaded_files))
				{
					foreach ($uploaded_files as $file)
					{
						$result = $this->misc_model->add_attachment_to_database($daily_log_id, 'job_daily_log', [$file]);
					}
				}

				// Send notification to customer if visible
				if ($data['is_visible_to_customer'] == 1)
				{
					$this->send_daily_log_notification($data['job_id'], $daily_log_id);
				}

				echo json_encode([
					'success' => TRUE,
					'message' => _l('daily_log_added'),
					'id'      => $daily_log_id,
				]);
			} else
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_failed_to_add'),
				]);
			}
			die();
		}
	}

	/**
	 * Edit daily log
	 *
	 * @param  int  $id  daily log id
	 */
	public function edit_daily_log($id)
	{
		if ($this->input->post())
		{
			$data = $this->input->post();

			// Check if can edit
			if ( ! $this->job_daily_logs_model->can_edit($id, get_staff_user_id()))
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_cannot_edit'),
				]);
				die();
			}

			// Validate required fields
			if (empty($data['comments']))
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_comments_required'),
				]);
				die();
			}

			// HTML purify comments
			hooks()->add_filter('html_purify_content', function ($will_purify) {
				return FALSE;
			});
			$data['comments'] = html_purify($this->input->post('comments', FALSE));

			$success = $this->job_daily_logs_model->update($id, $data);

			if ($success)
			{
				// Handle photo uploads
				$log            = $this->job_daily_logs_model->get($id);
				$uploaded_files = handle_job_daily_log_attachments($log->job_id, $id);
				if ($uploaded_files && is_array($uploaded_files))
				{
					foreach ($uploaded_files as $file)
					{
						$this->misc_model->add_attachment_to_database($id, 'job_daily_log', [$file]);
					}
				}

				echo json_encode([
					'success' => TRUE,
					'message' => _l('daily_log_updated'),
				]);
			} else
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_failed_to_update'),
				]);
			}
			die();
		}

		// Show edit form
		if (staff_cant('view', 'tasks'))
		{
			access_denied('jobs');
		}

		$log = $this->job_daily_logs_model->get($id);

		if ( ! $log)
		{
			show_404();
		}

		// Check if can edit
		if ( ! $this->job_daily_logs_model->can_edit($id, get_staff_user_id()))
		{
			access_denied('daily_logs');
		}

		$job = $this->jobs_model->get($log->job_id);

		$data['log']   = $log;
		$data['job']   = $job;
		$data['title'] = _l('edit_daily_log').' - '.$job->job_title;

		$this->load->view('jobs/daily_logs/edit', $data);
	}

	/**
	 * Delete daily log
	 *
	 * @param  int  $id  daily log id
	 */
	public function delete_daily_log($id)
	{
		// Check if can delete
		if ( ! $this->job_daily_logs_model->can_delete($id, get_staff_user_id()))
		{
			set_alert('danger', _l('daily_log_cannot_delete'));
			redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
			die();
		}

		$log = $this->job_daily_logs_model->get($id);

		if ( ! $log)
		{
			show_404();
		}

		$success = $this->job_daily_logs_model->delete($id);

		if ($success)
		{
			set_alert('success', _l('daily_log_deleted'));
		} else
		{
			set_alert('danger', _l('daily_log_failed_to_delete'));
		}

		redirect(admin_url('trade_job_management/jobs/daily_logs/'.$log->job_id));
	}

	public function delete_daily_log_photo($id)
	{
		if ($this->input->post())
		{
			$log = $this->job_daily_logs_model->get_by_photo_id($id);

			if ( ! $log || ! $this->job_daily_logs_model->can_delete($log->daily_log_id, get_staff_user_id()))
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_cannot_delete'),
				]);
				die();
			}

			$success = $this->job_daily_logs_model->delete_photo($log->job_id, $id);

			if ($success)
			{
				echo json_encode([
					'success' => TRUE,
					'message' => _l('daily_log_photo_deleted'),
				]);
			} else
			{
				echo json_encode([
					'success' => FALSE,
					'message' => _l('daily_log_photo_failed_to_delete'),
				]);
			}
			die();
		}
	}

	/**
	 * Get daily logs data via AJAX
	 *
	 * @param  int  $job_id  job id
	 */
	public function get_daily_logs_data($job_id)
	{
		if ($this->input->is_ajax_request())
		{
			$date = $this->input->get('date');

			if ($date)
			{
				$logs = $this->job_daily_logs_model->get_by_date($job_id, $date);
			} else
			{
				$logs = $this->job_daily_logs_model->get_by_job_id($job_id);
			}

			echo json_encode([
				'success' => TRUE,
				'logs'    => $logs,
			]);
			die();
		}
	}

	/**
	 * Send daily log notification to customer
	 *
	 * @param  int  $job_id  job id
	 * @param  int  $daily_log_id  daily log id
	 */
	private function send_daily_log_notification($job_id, $daily_log_id)
	{
		$job = $this->jobs_model->get($job_id);

		if ( ! $job || ! $job->client)
		{
			return FALSE;
		}

		// Get primary contact
		$this->load->model('clients_model');
		$contacts = $this->clients_model->get_contacts($job->client->userid);

		if (empty($contacts))
		{
			return FALSE;
		}

		// Send to primary contact
		foreach ($contacts as $contact)
		{
			if ($contact['is_primary'] == 1)
			{
				$this->send_daily_log_email($contact['email'], $job_id, $daily_log_id, $contact['id']);
				break;
			}
		}

		// If no primary contact, send to first contact
		if (empty($contact['is_primary']))
		{
			$this->send_daily_log_email($contacts[0]['email'], $job_id, $daily_log_id, $contacts[0]['id']);
		}

		return TRUE;
	}

	/**
	 * Send daily log email
	 *
	 * @param  string  $email  recipient email
	 * @param  int  $job_id  job id
	 * @param  int  $daily_log_id  daily log id
	 * @param  int  $contact_id  contact id
	 */
	private function send_daily_log_email($email, $job_id, $daily_log_id, $contact_id)
	{
		$path = APP_MODULES_PATH.TRADE_JOB_MANAGEMENT_MODULE_NAME.'/libraries/mails/Job_daily_log_added.php';

		if ( ! file_exists($path))
		{
			return FALSE;
		}

		if ( ! class_exists('Job_daily_log_added', FALSE))
		{
			include_once $path;
		}

		$mail = new Job_daily_log_added($email, $job_id, $daily_log_id, $contact_id);

		return $mail->send();
	}

}
