<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Job_timers_model extends App_Model {

	/**
	 * Timer action, START/STOP Timer
	 *
	 * @param  mixed  $job_id  job id
	 * @param  mixed  $timer_id  timer_id to stop the timer
	 * @param  string  $note  note for timer
	 * @param  boolean  $admin_stop  is admin want to stop timer from another staff member
	 *
	 * @return boolean
	 */
	public function timer_tracking($job_id = '', $timer_id = '', $note = '', $cost = 0, $admin_stop = FALSE)
	{
		if ($job_id == '' && $timer_id == '')
		{
			return FALSE;
		}

		if ($job_id !== '0' && ! $admin_stop)
		{
			if ( ! $this->is_job_creator(get_staff_user_id(), $job_id) && ! $this->is_job_specialist(get_staff_user_id(), $job_id))
			{
				return FALSE;
			} elseif ($this->is_job_billed($job_id))
			{
				return FALSE;
			}
		}

		$timer = $this->get_job_timer([
			'timerid' => $timer_id,
		]);

		$new_timer = FALSE;

		if ($timer == NULL)
		{
			$new_timer = TRUE;
		}

		if ($new_timer)
		{
			// Stop all other timesheets when adding new timesheet
			$this->db->where('job_id', $job_id);
			$this->db->where('staff_id', get_staff_user_id());
			$this->db->where('end_time IS NULL');
			$this->db->update(db_prefix().'tjm_job_timers', [
				'end_time' => time(),
			]);

			$this->db->select('hourly_rate');
			$this->db->from(db_prefix().'staff');
			$this->db->where('staffid', get_staff_user_id());
			$hourly_rate = $this->db->get()->row()->hourly_rate;
			$this->db->insert(db_prefix().'tjm_job_timers', [
				'start_time'  => time(),
				'staff_id'    => get_staff_user_id(),
				'job_id'      => $job_id,
				'hourly_rate' => $hourly_rate,
				'cost'        => $cost,
				'total'       => $cost,
				'note'        => ($note != '' ? $note : NULL),
			]);

			$_new_timer_id = $this->db->insert_id();

			hooks()->do_action('job_timer_started', ['job_id' => $job_id, 'timer_id' => $_new_timer_id]);

			return TRUE;
		}

		if ($timer)
		{
			if ($timer->end_time != NULL)
			{
				return FALSE;
			}

			$end_time = hooks()->apply_filters('before_job_timer_stopped', time(), [
				'timer'  => $timer,
				'job_id' => $job_id,
				'note'   => $note,
			]);

			$duration = intval($end_time) - intval($timer->start_time);
			if ($duration < 0)
			{
				$total_hours = 0;
			} else
			{
				$total_hours = floatval($duration) / (60.0 * 60.0);
			}
			$total = $cost + $total_hours * floatval($timer->hourly_rate);

			$this->db->where('timerid', $timer_id);
			$this->db->update(db_prefix().'tjm_job_timers', [
				'end_time' => $end_time,
				'job_id'   => $job_id,
				'cost'     => $cost,
				'total'    => $total,
				'note'     => ($note != '' ? $note : NULL),
			]);
		}

		return TRUE;
	}

	public function timesheet($data)
	{
		$staff_id = get_staff_user_id();
		if (isset($data['staff_id']) && $data['staff_id'] != '')
		{
			$staff_id = $data['staff_id'];
		}

		$timer_id   = $data['timer_id'] ?? '';
		$start_time = $data['start_time'] ?? time();
		$start_time = strtotime($start_time);
		$end_time   = $data['end_time'] ?? NULL;
		if ( ! empty($end_time))
		{
			$end_time = strtotime($end_time);
		}
		$job_id   = $data['job_id'];
		$cost     = $data['cost'] ?? 0;
		$note     = nl2br($data['note'] ?? '');
		$duration = intval($end_time) - intval($start_time);
		if ($duration < 0)
		{
			$total_hours = 0;
		} else
		{
			$total_hours = floatval($duration) / (60.0 * 60.0);
		}

		$args = [
			'start_time' => $start_time,
			'end_time'   => $end_time,
			'staff_id'   => $staff_id,
			'job_id'     => $job_id,
			'cost'       => $cost,
			'note'       => $note,
		];

		if (empty($timer_id))
		{
			// Stop all other timesheets when adding new timesheet
			$this->db->where('job_id', $job_id);
			$this->db->where('staff_id', $staff_id);
			$this->db->where('end_time IS NULL');
			$this->db->update(db_prefix().'tjm_job_timers', [
				'end_time' => time(),
			]);

			$this->db->select('hourly_rate');
			$this->db->from(db_prefix().'staff');
			$this->db->where('staffid', $staff_id);
			$hourly_rate = $this->db->get()->row()->hourly_rate;

			$args['hourly_rate'] = $hourly_rate;
			$args['total']       = $cost + $total_hours * $hourly_rate;

			$this->db->insert(db_prefix().'tjm_job_timers', $args);

			return $this->db->insert_id() > 0;
		}

		$this->db->where('timerid', $timer_id);
		$timer = $this->db->get(db_prefix().'tjm_job_timers')->row_array();
		if (empty($timer))
		{
			return FALSE;
		}

		$hourly_rate = floatval($timer['hourly_rate']);

		$args['total'] = $cost + $hourly_rate * $total_hours;

		$affectedRows = 0;
		$this->db->where('timerid', $data['timer_id']);
		$this->db->update(db_prefix().'tjm_job_timers', $args);

		if ($this->db->affected_rows() > 0)
		{
			$affectedRows++;
		}

		return $affectedRows > 0;
	}

	public function get_timers($job_id, $where = [])
	{
		$this->db->where($where);
		$this->db->where('job_id', $job_id);
		$this->db->order_by('start_time', 'DESC');

		return $this->db->get(db_prefix().'tjm_job_timers')->result_array();
	}

	public function get_job_timer($where)
	{
		$this->db->where($where);

		return $this->db->get(db_prefix().'tjm_job_timers')->row();
	}

	public function is_timer_started($job_id, $staff_id = '')
	{
		if ($staff_id == '')
		{
			$staff_id = get_staff_user_id();
		}

		$timer = $this->get_last_timer($job_id, $staff_id);

		if ( ! $timer || $timer->end_time != NULL)
		{
			return FALSE;
		}

		return $timer;
	}

	public function is_timer_started_for_job($id, $where = [])
	{
		$this->db->where('job_id', $id);
		$this->db->where('end_time IS NULL');
		$this->db->where($where);
		$results = $this->db->count_all_results(db_prefix().'tjm_job_timers');

		return $results > 0;
	}

	public function get_last_timer($job_id, $staff_id = '')
	{
		if ($staff_id == '')
		{
			$staff_id = get_staff_user_id();
		}
		$this->db->where('staff_id', $staff_id);
		$this->db->where('job_id', $job_id);
		$this->db->order_by('timerid', 'desc');
		$this->db->limit(1);

		return $this->db->get(db_prefix().'tjm_job_timers')->row();
	}

	public function delete_timesheet($id)
	{
		$this->db->where('timerid', $id);
		$timesheet = $this->db->get(db_prefix().'tjm_job_timers')->row();
		$this->db->where('timerid', $id);
		$this->db->delete(db_prefix().'tjm_job_timers');
		if ($this->db->affected_rows() > 0)
		{
			hooks()->do_action('job_timer_deleted', $timesheet);

			log_activity('Timesheet Deleted ['.$id.']');

			return TRUE;
		}

		return FALSE;
	}

	public function get_staff_started_timers()
	{
		$this->db->select(db_prefix().'tjm_job_timers.*,'.db_prefix().'tjm_jobs.job_title');
		$this->db->join(db_prefix().'staff', db_prefix().'staff.staffid='.db_prefix().'tjm_job_timers.staff_id');
		$this->db->join(db_prefix().'tjm_jobs', db_prefix().'tjm_jobs.jobid='.db_prefix().'tjm_job_timers.job_id', 'left');
		$this->db->where('staff_id', get_staff_user_id());
		$this->db->where('end_time IS NULL');

		return $this->db->get(db_prefix().'tjm_job_timers')->result_array();
	}

	/**
	 * Check is user is job creator
	 *
	 * @param  mixed  $user_id  staff id
	 * @param  mixed  $job_id  job id
	 *
	 * @return boolean
	 */
	public function is_job_creator($user_id, $job_id)
	{
		return total_rows(db_prefix().'tjm_jobs', [
				'created_by' => $user_id,
				'jobid'      => $job_id,
			]) > 0;
	}

	/**
	 * Check is user is job specialist
	 *
	 * @param  mixed  $user_id  staff id
	 * @param  mixed  $job_id  job id
	 *
	 * @return boolean
	 */
	public function is_job_specialist($user_id, $job_id): bool
	{
		return total_rows(db_prefix().'tjm_job_specialists', [
				'staff_id' => $user_id,
				'job_id'   => $job_id,
			]) > 0;
	}

	public function is_job_billed($id)
	{
		return total_rows(db_prefix().'tjm_jobs', [
				'jobid'  => $id,
				'billed' => 1,
			]) > 0;
	}

}