<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Properties extends AdminController
{
    protected const BASE_PATH = '/properties';

    public function __construct()
    {
        parent::__construct();

        $this->load->model('properties_model');
        $this->load->model('property_types_model');
        $this->load->model('landlords_model');
        $this->load->model('jobs_model');
        $this->load->model('misc_model');
        $this->load->helper('date_helper');
    }

    /* List all clients */

    public function index()
    {
        if (staff_cant('view', 'projects')) {
            if (!have_assigned_customers() && staff_cant('create', 'projects')) {
                access_denied('projects');
            }
        }

        $data['title'] = _l('properties');
        $data['table'] = App_table::find('properties');

        $this->load->view(static::BASE_PATH . '/index', $data);
    }

    public function view($id = '', $tab = 'overview', $jobId = '')
    {
        if (staff_can('view', 'projects') || $this->properties_model->is_member($id)) {
            $property = $this->properties_model->get($id);

            if (!$property) {
                blank_page(_l('project_not_found'));
            }

            $group = !$this->input->get('group') ? 'project_overview' : $this->input->get('group');

            if (strpos($group, '#') !== false) {
                $group = str_replace('#', '', $group);
            }

            $data['tabs'] = get_project_tabs_admin();
            $data['tab'] = $this->app_tabs->filter_tab($data['tabs'], $group);

            if (!$data['tab']) {
                show_404();
            }

            $data['property'] = $property;

            $data['staff'] = $this->staff_model->get('', ['active' => 1]);
            $data['members'] = $this->properties_model->get_property_managements($id);
            $data['title'] = $data['property']->name;
            $data['project_status'] = get_project_status_by_id($property->status);
            $data['current_tab'] = empty($tab) ? 'overview' : $tab;
            $data['job_id'] = empty($jobId) ? null : $jobId;
            $data['property_attachments'] = $this->properties_model->get_property_attachments($property->propertyid);

            $this->load->view('properties/view', $data);
        } else {
            access_denied('Property View');
        }
    }

    public function add_photos($id)
    {
        if (staff_cant('edit', 'projects')) {
            access_denied('Projects');
        }

        $property = $this->properties_model->get($id);
        if (!$property) {
            show_404();
        }
        if (empty($this->input->post()) && empty($_FILES ?? [])) {
            redirect(admin_url('trade_job_management' . static::BASE_PATH));
            exit();
        }
        if (!empty($_FILES ?? [])) {
            $uploaded_files = tjm_handle_property_attachments_array($id);
            if ($uploaded_files && is_array($uploaded_files)) {
                foreach ($uploaded_files as $file) {
                    $this->misc_model->add_attachment_to_database($id, 'property', [$file]);
                }
            }

            set_alert('success', _l('tjm_property_photos_uploaded_successfully'));
            redirect(admin_url('trade_job_management' . static::BASE_PATH . '/view/' . $id));
        }
        set_alert('error', _l('tjm_property_photos_select_photos'));
        redirect(admin_url('trade_job_management' . static::BASE_PATH . '/view/' . $id));
    }

    public function download($key)
    {
        if (!is_staff_logged_in()) {
            access_denied('Properties');
        }
        if (!$key) {
            show_404();
        }
        $this->db->where('attachment_key', $key);
        $attachment = $this->db->get(db_prefix() . 'files')->row();
        if (!$attachment) {
            show_404();
        }
        $path = get_upload_path_by_type('property') . $attachment->rel_id . '/' . $attachment->file_name;
        tjm_force_download($path, null);
    }

    function delete_attachment($property_id, $attachment_id)
    {
        if (!is_staff_logged_in()) {
            access_denied('Properties');
        }
        $this->db->where('id', $attachment_id)->where('rel_type', 'property')->where('rel_id', $property_id);
        $attachment = $this->db->get(db_prefix() . 'files')->row();
        if (!$attachment) {
            show_404();
        }
        if ($attachment->staff_id !== get_staff_user_id() && !is_admin()) {
            access_denied('Properties');
        }

        $this->db->where('id', $attachment_id)->delete(db_prefix() . 'files');

        set_alert('success', _l('tjm_photo_deleted_successfully'));
        redirect(admin_url('trade_job_management' . static::BASE_PATH . '/view/' . $property_id));
    }

    /* Edit client or add new client*/

    public function delete($project_id)
    {
        if (staff_can('delete', 'projects')) {
            $success = $this->properties_model->delete($project_id);
            if ($success) {
                set_alert('success', _l('deleted', _l('property')));
                redirect(admin_url('trade_job_management' . static::BASE_PATH));
            } else {
                set_alert('warning', _l('problem_deleting', _l('property_lowercase')));
                redirect(admin_url('trade_job_management' . static::BASE_PATH . '/view/' . $project_id));
            }
        }
    }

    public function table($clientid = '')
    {
        App_table::find('properties')->output([
            'clientid' => $clientid,
        ]);
    }

    /* Edit client or add new client*/

    public function create($landlord_id = '')
    {
        if (staff_cant('create', 'projects')) {
            access_denied('Properties');
        }

        if ($this->input->post()) {
            $data = $this->input->post();
            $data['description'] = html_purify($this->input->post('description', false));
            if (staff_cant('create', 'projects')) {
                access_denied('Properties');
            }

            $id = $this->properties_model->add($data);
            if ($id) {
                $uploaded_files = tjm_handle_property_attachments_array($id);
                if ($uploaded_files && is_array($uploaded_files)) {
                    foreach ($uploaded_files as $file) {
                        $this->misc_model->add_attachment_to_database($id, 'project', [$file]);
                    }
                }

                set_alert('success', _l('added_successfully', _l('property')));
                redirect(admin_url('trade_job_management' . static::BASE_PATH . '/view/' . $id));
            }
        }

        $data['title'] = _l('add_new', _l('property_lowercase'));
        ;
        $data['property_types'] = $this->property_types_model->get('', true);
        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        $data['landlord_id'] = $landlord_id;

        $this->load->view('properties/create', $data);
    }

    public function edit($id = '')
    {
        if (staff_cant('edit', 'projects')) {
            access_denied('Projects');
        }

        $property = $this->properties_model->get($id);

        if ($this->input->post()) {
            $data = $this->input->post();
            $data['description'] = html_purify($this->input->post('description', false));
            if (staff_cant('edit', 'projects')) {
                access_denied('Projects');
            }
            $success = $this->properties_model->update($data, $id);
            if ($success) {
                set_alert('success', _l('updated_successfully', _l('property')));
            }
            redirect(admin_url('trade_job_management' . static::BASE_PATH . '/view/' . $id));
        }

        $data['property'] = $property;
        $data['property_members'] = $this->properties_model->get_property_managements($id);
        $data['title'] = _l('edit', _l('property'));
        $data['customer_id'] = $property->clientid;
        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        $data['property_types'] = $this->property_types_model->get('', true);

        $this->load->view('properties/edit', $data);
    }

    public function jobs_table($property_id)
    {
        App_table::find('jobs')->output(['property_id' => $property_id]);
    }

    public function get_rel_property_data($id, $job_id = '')
    {
        if ($this->input->is_ajax_request()) {
            $selected_milestone = '';
            $assigned = '';
            if ($job_id != '' && $job_id != 'undefined') {
                $task = $this->jobs_model->get($job_id);
                $assigned = array_map(function ($member) {
                    return $member['assigneeid'];
                }, $this->jobs_model->get_task_assignees($job_id));
            }

            echo
                json_encode([
                    'deadline' => $deadline,
                    'deadline_formatted' => $deadline ? _d($deadline) : null,
                    'allow_to_view_tasks' => $allow_to_view_tasks,
                    'billing_type' => get_project_billing_type($id),
                    'milestones' => render_select(
                        'milestone',
                        $this->projects_model->get_milestones($id),
                        [
                            'id',
                            'name',
                        ],
                        'task_milestone',
                        $selected_milestone,
                    ),
                    'assignees' => render_select(
                        'assignees[]',
                        $this->projects_model->get_project_members($id, true),
                        [
                            'staff_id',
                            ['firstname', 'lastname'],
                        ],
                        'task_single_assignees',
                        $assigned,
                        ['multiple' => true],
                        [],
                        '',
                        '',
                        false,
                    ),
                ])
            ;
        }
    }
}
