<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Jobs_model extends App_Model {

	const STATUS_REPORTED = 1;

	const STATUS_QUOTE_SENT = 2;

	const STATUS_SCHEDULED = 3;

	const STATUS_COMPLETE = 4;

	const STATUS_EXCEED_NTE = 5;

	public function get_statuses()
	{
		$statuses = hooks()->apply_filters('before_get_job_statuses', [
			[
				'id'             => static::STATUS_REPORTED,
				'color'          => '#64748b',
				'name'           => _l('job_status_reported'),
				'order'          => 1,
				'filter_default' => TRUE,
			],
			[
				'id'             => static::STATUS_SCHEDULED,
				'color'          => '#3b82f6',
				'name'           => _l('job_status_scheduled'),
				'order'          => 2,
				'filter_default' => TRUE,
			],
			[
				'id'             => static::STATUS_QUOTE_SENT,
				'color'          => '#84cc16',
				'name'           => _l('job_status_quote_sent'),
				'order'          => 4,
				'filter_default' => TRUE,
			],
			[
				'id'             => static::STATUS_COMPLETE,
				'color'          => '#22c55e',
				'name'           => _l('job_status_completed'),
				'order'          => 100,
				'filter_default' => FALSE,
			],
			[
				'id'             => static::STATUS_EXCEED_NTE,
				'color'          => '#ef4444',
				'name'           => _l('job_status_exceed_nte'),
				'order'          => 100,
				'filter_default' => TRUE,
			],
		]);

		usort($statuses, function ($a, $b) {
			return $a['order'] - $b['order'];
		});

		return $statuses;
	}

	/**
	 * Get job by id
	 *
	 * @param  mixed  $id  job id
	 *
	 * @return object
	 */
	public function get($id, $where = [])
	{
		$this->db->select(db_prefix().'tjm_jobs.*,type_name as job_type_name');
		$this->db->where('jobid', $id);
		$this->db->where($where);
		$this->db->join(db_prefix().'tjm_job_types', db_prefix().'tjm_job_types.jobtypeid = '.db_prefix().'tjm_jobs.job_type_id');
		$job = $this->db->get(db_prefix().'tjm_jobs')->row();
		if ($job)
		{
			$specialists         = $this->get_job_specialists($id);
			$job->comments       = $this->get_job_comments($id);
			$job->specialists    = $specialists;
			$job->specialist_ids = array_map(function ($specialist) {
				return $specialist['staff_id'];
			}, $specialists);

			$job->attachments = $this->get_job_attachments($id);

			if (is_staff_logged_in())
			{
				$job->current_user_is_specialist = $this->is_job_specialist(get_staff_user_id(), $id);
				$job->current_user_is_creator    = $this->is_job_creator(get_staff_user_id(), $id);
			}

			$job->property   = $this->get_job_property($job->property_id);
			$job->landlord   = $this->get_job_landlord($job->property['landlord_id']);
			$job->schedulers = $this->get_job_schedulers($job->jobid);

			$client_id = $job->landlord['client_id'];
			if ($client_id)
			{
				$job->clientid = $client_id;
				$job->client   = $this->clients_model->get($client_id);
			} else
			{
				$job->clientid = NULL;
				$job->client   = NULL;
			}
			$job->timesheets = $this->get_timesheets($id);
		}

		return hooks()->apply_filters('get_job', $job);
	}

	public function add($data, $clientRequest = FALSE)
	{
		$data['status'] = self::STATUS_REPORTED;

		$specialists = $data['specialists'];
		unset($data['specialists']);

		$data['created_at'] = date('Y-m-d H:i:s');
		if ( ! isset($data['created_by']))
		{
			$data['created_by'] = is_staff_logged_in() ? get_staff_user_id() : NULL;
		}

		hooks()->do_action('before_add_job', $data);

		$this->db->insert(db_prefix().'tjm_jobs', $data);
		$insert_id = $this->db->insert_id();
		if ($insert_id)
		{
			$job      = $this->db->get_where(db_prefix().'tjm_jobs', ['jobid' => $insert_id], 1)->row_array();
			$job_hash = generate_job_hash($job);
			$this->db->update(db_prefix().'tjm_jobs', ['job_hash' => $job_hash], ['jobid' => $insert_id]);

			if ( ! empty($specialists))
			{
				foreach ($specialists as $staff_id)
				{
					$this->add_job_specialists([
						'job_id'   => $insert_id,
						'staff_id' => $staff_id,
					]);
				}
			}

			log_activity('New Job Added [ID:'.$insert_id.', Name: '.$data['name'].']');
			hooks()->do_action('after_add_job', $insert_id);

			return $insert_id;
		}

		return FALSE;
	}

	public function update($data, $id)
	{
		$job_specialists = $this->get_job_specialists($id);
		foreach ($job_specialists as $specialist)
		{
			$this->remove_specialist($specialist['staff_id'], $id);
		}
		$specialists = $data['specialists'];
		unset($data['specialists']);
		if ( ! empty($specialists))
		{
			foreach ($specialists as $staff_id)
			{
				$this->add_job_specialists([
					'job_id'   => $id,
					'staff_id' => $staff_id,
				]);
			}
		}

		$this->db->where('jobid', $id);
		$this->db->update(db_prefix().'tjm_jobs', $data);

		return $this->db->affected_rows() > 0;
	}

	public function update_invoice($id, $invoice_id)
	{
		$this->db->where('jobid', $id);
		$this->db->where('invoice_id');
		$this->db->update(db_prefix().'tjm_jobs', [
			'invoice_id' => $invoice_id,
			'billed'     => TRUE,
		]);

		return $this->db->affected_rows() > 0;
	}

	/**
	 * Get all job assigneed
	 *
	 * @param  mixed  $id  job id
	 *
	 * @return array
	 */
	public function get_job_specialists($id): array
	{
		$this->db->select('jsid,'.db_prefix().'tjm_job_specialists.staff_id as staff_id,firstname,lastname,CONCAT(firstname, " ", lastname) as full_name,phonenumber');
		$this->db->from(db_prefix().'tjm_job_specialists');
		$this->db->join(db_prefix().'staff', db_prefix().'staff.staffid = '.db_prefix().'tjm_job_specialists.staff_id');
		$this->db->where('job_id', $id);
		$this->db->order_by('firstname', 'asc');

		return $this->db->get()->result_array();
	}

	/**
	 * Change job status
	 *
	 * @param  mixed  $status  job status
	 * @param  mixed  $job_id  job id
	 *
	 * @return boolean
	 */
	public function mark_as($status, $job_id)
	{
		$this->db->where('jobid', $job_id);
		$job = $this->db->get(db_prefix().'tjm_jobs')->row();

		if ($job->status == static::STATUS_COMPLETE)
		{
			return $this->unmark_complete($job_id, $status);
		}

		// Validate client confirmation requirement before marking as complete
		if ($status == static::STATUS_COMPLETE)
		{
			// Get property information to check if confirmation is required
			$this->db->select('require_client_confirmation');
			$this->db->where('propertyid', $job->property_id);
			$property = $this->db->get(db_prefix().'tjm_properties')->row();

			// Check if property requires client confirmation and job hasn't been confirmed
			if ($property && $property->require_client_confirmation == 1 && empty($job->confirmation_at))
			{
				// Set error message for user feedback
				$CI = &get_instance();
				$CI->session->set_flashdata('danger-alert', _l('job_requires_client_confirmation'));

				return FALSE;
			}
		}

		$update = ['status' => $status];

		if ($status == static::STATUS_COMPLETE)
		{
			$update['completed_at'] = date('Y-m-d H:i:s');
		}

		$this->db->where('jobid', $job_id);
		$this->db->update(db_prefix().'tjm_jobs', $update);
		if ($this->db->affected_rows() > 0)
		{
			hooks()->do_action('job_status_changed', ['status' => $status, 'job_id' => $job_id]);

			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Unmark job as complete
	 *
	 * @param  mixed  $id  job id
	 *
	 * @return boolean
	 */
	public function unmark_complete($id, $force_to_status = FALSE): bool
	{
		if ($force_to_status != FALSE)
		{
			$status = $force_to_status;
		} else
		{
			$status = 1;
			$this->db->select('created_at');
			$this->db->where('jobid', $id);
			$job = $this->db->get(db_prefix().'tjm_jobs')->row();
			if (date('Y-m-d') > date('Y-m-d', strtotime($job->created_at)))
			{
				$status = 4;
			}
		}

		$this->db->where('jobid', $id);
		$this->db->update(db_prefix().'tjm_jobs', [
			'completed_at' => NULL,
			'status'       => $status,
		]);

		if ($this->db->affected_rows() > 0)
		{
			hooks()->do_action('job_status_changed', ['status' => $status, 'job_id' => $id]);

			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Send notification on job activity to creator,follower/s,assignee/s
	 *
	 * @param  string  $description  notification description
	 * @param  mixed  $jobid  job id
	 * @param  boolean  $excludeid  excluded staff id to not send the notifications
	 *
	 * @return boolean
	 */
	private function _send_job_responsible_users_notification($description, $jobid, $excludeid = FALSE, $email_template = '', $additional_notification_data = '', $comment_id = FALSE)
	{
		$this->load->model('staff_model');

		$staff = $this->staff_model->get('', ['active' => 1]);

		$notifiedUsers = [];
		foreach ($staff as $member)
		{
			if (is_numeric($excludeid))
			{
				if ($excludeid == $member['staffid'])
				{
					continue;
				}
			}
			if ( ! is_client_logged_in())
			{
				if ($member['staffid'] == get_staff_user_id())
				{
					continue;
				}
			}

			if ($this->should_staff_receive_notification($member['staffid'], $jobid))
			{
				$link = '#jobid='.$jobid;

				if ($comment_id)
				{
					$link .= '#comment_'.$comment_id;
				}

				$notified = add_notification([
					'description'     => $description,
					'touserid'        => $member['staffid'],
					'link'            => $link,
					'additional_data' => $additional_notification_data,
				]);

				if ($notified)
				{
					array_push($notifiedUsers, $member['staffid']);
				}

				if ($email_template != '')
				{
					$this->_send_mail_template($email_template, $member['email'], $member['staffid'], $jobid);
				}
			}
		}

		pusher_trigger_notification($notifiedUsers);
	}

	/**
	 * Check whether the given staff should receive notification for
	 * the given job
	 *
	 * @param  int  $staffid
	 * @param  int  $jobid  [description]
	 *
	 * @return boolean
	 */
	private function should_staff_receive_notification($staffid, $jobid)
	{
		if ( ! $this->can_staff_access_job($staffid, $jobid))
		{
			return FALSE;
		}

		return hooks()->apply_filters(
			'should_staff_receive_job_notification',
			$this->is_job_specialist($staffid, $jobid) || $this->is_job_follower($staffid, $jobid) || $this->is_job_creator($staffid, $jobid) || $this->staff_has_commented_on_job($staffid, $jobid),
			['staff_id' => $staffid, 'job_id' => $jobid],
		);
	}

	/**
	 * Send mail template
	 *
	 * @return mixed
	 * @since  2.3.0
	 */
	private function _send_mail_template()
	{
		$params = func_get_args();

		// First params is the $class param
		$class = $params[0];
		$class = ucfirst($class);

		unset($params[0]);

		$params = array_values($params);

		$path = APP_MODULES_PATH.TRADE_JOB_MANAGEMENT_MODULE_NAME.'/libraries/mails/'.$class.'.php';

		if ( ! file_exists($path))
		{
			if ( ! defined('CRON'))
			{
				show_error('Mail Class Does Not Exists ['.$path.']');
			} else
			{
				return FALSE;
			}
		}

		// Include the mailable class
		if ( ! class_exists($class, FALSE))
		{
			include_once $path;
		}

		// Initialize the class and pass the params
		$instance = new $class(...$params);

		return $instance->send();
	}

	public function _send_customer_contacts_notification($jobid, $template_name)
	{
		$this->db->select('rel_id,visible_to_client,rel_type');
		$this->db->from(db_prefix().'jobs');
		$this->db->where('id', $jobid);
		$job = $this->db->get()->row();

		if ($job->rel_type == 'project')
		{
			$this->db->where('project_id', $job->rel_id);
			$this->db->where('name', 'view_jobs');
			$project_settings = $this->db->get(db_prefix().'project_settings')->row();

			if ($project_settings)
			{
				if ($project_settings->value == 1 && $job->visible_to_client == 1)
				{
					$this->load->model('landlords_model');
					$contacts = $this->landlords_model->get_contacts_for_project_notifications($project_settings->project_id, 'job_emails');
					foreach ($contacts as $contact)
					{
						if (is_client_logged_in() && get_contact_user_id() == $contact['id'])
						{
							continue;
						}
						$this->_send_mail_template($template_name, $contact['email'], $contact['userid'], $contact['id'], $jobid);
					}
				}
			}
		}
	}

	public function get_billable_job_data($job_id)
	{
		$this->db->where('id', $job_id);
		$data = $this->db->get(db_prefix().'jobs')->row();
		if ($data->rel_type == 'project')
		{
			$this->db->select('billing_type,project_rate_per_hour,name');
			$this->db->where('id', $data->rel_id);
			$project      = $this->db->get(db_prefix().'projects')->row();
			$billing_type = get_project_billing_type($data->rel_id);

			if ($project->billing_type == 2)
			{
				$data->hourly_rate = $project->project_rate_per_hour;
			}

			$data->name = $project->name.' - '.$data->name;
		}
		$data->total_hours   = 1;
		$data->total_seconds = 0;
		$data->description   = '';
		$data->quantity      = 1;

		return $data;
	}

	/**
	 * Check is user is job creator
	 *
	 * @param  mixed  $user_id  staff id
	 * @param  mixed  $job_id  job id
	 *
	 * @return boolean
	 */
	public function is_job_creator($user_id, $job_id)
	{
		return total_rows(db_prefix().'tjm_jobs', [
				'created_by' => $user_id,
				'jobid'      => $job_id,
			]) > 0;
	}

	/**
	 * Check is user is job specialist
	 *
	 * @param  mixed  $user_id  staff id
	 * @param  mixed  $job_id  job id
	 *
	 * @return boolean
	 */
	public function is_job_specialist($user_id, $job_id): bool
	{
		return total_rows(db_prefix().'tjm_job_specialists', [
				'staff_id' => $user_id,
				'job_id'   => $job_id,
			]) > 0;
	}

	public function is_job_billed($id)
	{
		return total_rows(db_prefix().'tjm_jobs', [
				'id'     => $id,
				'billed' => 1,
			]) > 0;
	}

	/**
	 * Get job comment
	 *
	 * @param  mixed  $id  job id
	 *
	 * @return array
	 */
	public function get_job_comments($id): array
	{
		$this->db->select(
			'jcid,created_at,updated_at,content,'.db_prefix().'staff.firstname,'.db_prefix().'staff.lastname,'.db_prefix().'tjm_job_comments.staff_id,'.'file_id,CONCAT(firstname, " ", lastname) as staff_full_name',
		);
		$this->db->from(db_prefix().'tjm_job_comments');
		$this->db->join(db_prefix().'staff', db_prefix().'staff.staffid = '.db_prefix().'tjm_job_comments.staff_id', 'left');
		$this->db->where('job_id', $id);
		$this->db->order_by('created_at', 'desc');

		$comments = $this->db->get()->result_array();

		$ids = [];
		foreach ($comments as $key => $comment)
		{
			$ids[]                         = $comment['jcid'];
			$comments[$key]['attachments'] = [];
		}

		if (count($ids) > 0)
		{
			$allAttachments = $this->get_job_attachments($id, 'job_comment_id IN ('.implode(',', $ids).')');
			foreach ($comments as $key => $comment)
			{
				foreach ($allAttachments as $attachment)
				{
					if ($comment['jcid'] == $attachment['job_comment_id'])
					{
						$comments[$key]['attachments'][] = $attachment;
					}
				}
			}
		}

		return $comments;
	}

	/**
	 * Get all job attachments
	 *
	 * @param  mixed  $job_id  job id
	 *
	 * @return array
	 */
	private function get_job_attachments($job_id, $where = [])
	{
		$this->db->select(implode(', ', prefixed_table_fields_array(db_prefix().'files')).', '.db_prefix().'tjm_job_comments.jcid as comment_file_id');
		$this->db->where(db_prefix().'files.rel_id', $job_id);
		$this->db->where(db_prefix().'files.rel_type', 'job');

		if (is_array($where) && count($where) > 0 || is_string($where) && $where != '')
		{
			$this->db->where($where);
		}

		$this->db->join(db_prefix().'tjm_job_comments', db_prefix().'tjm_job_comments.file_id = '.db_prefix().'files.id', 'left');
		$this->db->join(db_prefix().'tjm_jobs', db_prefix().'tjm_jobs.jobid = '.db_prefix().'files.rel_id');
		$this->db->order_by(db_prefix().'files.dateadded', 'DESC');

		return $this->db->get(db_prefix().'files')->result_array();
	}

	private function get_job_property($id)
	{
		return $this->db->get_where(db_prefix().'tjm_properties', ['propertyid' => $id])->row_array();
	}

	private function get_job_landlord($id)
	{
		return $this->db->get_where(db_prefix().'tjm_landlords', ['landlordid' => $id])->row_array();
	}

	private function get_job_schedulers($id): array
	{
		$this->db->where('job_id', $id);
		$this->db->order_by('created_at', 'desc');

		return $this->db->get(db_prefix().'tjm_job_scheduled_logs')->result_array();
	}

	/**
	 * Add new job comment
	 *
	 * @param  array  $data  comment $_POST data
	 *
	 * @return boolean
	 */
	public function add_job_comment($data)
	{
		$this->db->insert(db_prefix().'tjm_job_comments', [
			'job_id'   => $data['jobid'],
			'content'  => $data['content'],
			'staff_id' => get_staff_user_id(),
		]);

		$insert_id = $this->db->insert_id();

		if ($insert_id)
		{
			$this->db->where('jobid', $data['jobid']);
			hooks()->do_action('job_comment_added', ['job_id' => $data['jobid'], 'comment_id' => $insert_id]);

			return $insert_id;
		}

		return FALSE;
	}

	/**
	 * Update job comment
	 *
	 * @param  array  $data  comment $_POST data
	 *
	 * @return bool
	 */
	public function edit_job_comment(array $data): bool
	{
		$this->db->update(db_prefix().'tjm_job_comments', [
			'content' => $data['content'],
		], [
			'jcid'   => $data['id'],
			'job_id' => $data['job_id'],
		]);

		$affected = $this->db->affected_rows();

		if ($affected > 0)
		{
			hooks()->do_action('job_comment_updated', [
				'job_id'     => $data['job_id'],
				'comment_id' => $data['id'],
			]);

			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Add uploaded attachments to database
	 *
	 * @since  Version 1.0.1
	 */
	public function add_attachment_to_database($rel_id, $attachment, $external = FALSE)
	{
		$file_id = $this->misc_model->add_attachment_to_database($rel_id, 'job', $attachment, $external);
		if ($file_id)
		{
			$file = $this->misc_model->get_file($file_id);
			$this->db->insert(db_prefix().'tjm_job_comments', [
				'content'  => '[job_attachment]',
				'job_id'   => $rel_id,
				'staff_id' => $file->staffid,
				'file_id'  => $file_id,
			]);

			return TRUE;
		}

		return FALSE;
	}

	public function update_attachment_job_comment_id($attachment_id, $job_comment_id)
	{
		$this->db->where('id', $attachment_id);
		$this->db->update(db_prefix().'files', [
			'job_comment_id' => $job_comment_id,
		]);
	}

	private function add_job_specialists($data)
	{
		$this->db->insert(db_prefix().'tjm_job_specialists', $data);
	}

	private function remove_specialist($staff_id, $id)
	{
		$this->db->where('job_id', $id);
		$this->db->where('staff_id', $staff_id);

		$this->db->delete(db_prefix().'tjm_job_specialists');
	}

	public function get_timesheets($job_id)
	{
		$job_id = $this->db->escape_str($job_id);

		return $this->db->query(
			"SELECT timerid,".db_prefix()."tjm_job_timers.note as note, start_time,end_time,cost,".db_prefix()."tjm_job_timers.hourly_rate as hourly_rate,total,job_id,staff_id, CONCAT(firstname, ' ', lastname) as full_name,
        end_time - start_time time_spent FROM ".db_prefix().'tjm_job_timers JOIN '.db_prefix().'staff ON '.db_prefix().'staff.staffid='.db_prefix()."tjm_job_timers.staff_id WHERE job_id = '$job_id' ORDER BY start_time DESC",
		)->result_array();
	}

	/**
	 * @param $job_id
	 *
	 * @return float
	 */
	function get_nte($job_id): float
	{
		$this->db->select('nte');
		$this->db->where('jobid', $job_id);

		$this->db->get(db_prefix().'tjm_jobs')->row_array();

		return floatval($result['nte'] ?? 0);
	}

	/**
	 * Delete job and all connections
	 *
	 * @param  mixed  $id  jobid
	 *
	 * @return boolean
	 */
	public function delete_job($id)
	{
		$this->db->where('jobid', $id);
		$this->db->delete(db_prefix().'tjm_jobs');
		if ($this->db->affected_rows() > 0)
		{
			$this->db->where('job_id', $id);
			$this->db->delete(db_prefix().'tjm_job_comments');

			$this->db->where('job_id', $id);
			$this->db->delete(db_prefix().'tjm_job_nte_logs');

			$this->db->where('job_id', $id);
			$this->db->delete(db_prefix().'tjm_job_scheduled_logs');

			$this->db->where('job_id', $id);
			$this->db->delete(db_prefix().'tjm_job_specialists');

			$this->db->where('job_id', $id);
			$this->db->delete(db_prefix().'tjm_job_status_logs');

			$this->db->where('job_id', $id);
			$this->db->delete(db_prefix().'tjm_job_timers');

			if (is_dir(get_upload_path_by_type('job').$id))
			{
				delete_dir(get_upload_path_by_type('job').$id);
			}

			hooks()->do_action('job_deleted', $id);

			return TRUE;
		}

		return FALSE;
	}

}
