<?php

defined('BASEPATH') or exit('No direct script access allowed');

/** @property Jobs_model $jobs_model */
/** @property Job_daily_logs_model job_daily_logs_model */
class Job extends ClientsController {

	public function __construct()
	{
		parent::__construct();
		$this->load->model('jobs_model');
		$this->load->model('job_daily_logs_model');
	}

	/**
	 * Send notifications to staff when client confirms job
	 *
	 * @param int $job_id Job ID
	 * @return void
	 */
	private function send_job_confirmation_notifications($job_id)
	{
		$this->load->model('properties_model');

		$job = $this->jobs_model->get($job_id);
		if ( ! $job)
		{
			return;
		}

		$notifiedUsers = [];

		// Get all staff members who should be notified
		$staff_to_notify = [];

		// 1. Add job specialists
		foreach ($job->specialists as $specialist)
		{
			$staff_to_notify[$specialist['staff_id']] = $specialist['staff_id'];
		}

		// 2. Add job creator
		if ($job->created_by)
		{
			$staff_to_notify[$job->created_by] = $job->created_by;
		}

		// 3. Add property management team
		$property_managers = $this->properties_model->get_property_managements($job->property_id);
		foreach ($property_managers as $manager)
		{
			$staff_to_notify[$manager['staff_id']] = $manager['staff_id'];
		}

		// Send notification to each staff member
		$this->load->model('staff_model');
		foreach ($staff_to_notify as $staff_id)
		{
			$notified = add_notification([
				'description'     => 'not_job_client_confirmed',
				'touserid'        => $staff_id,
				'link'            => 'trade_job_management/jobs/view/'.$job_id,
				'additional_data' => serialize([
					$job->job_title,
					$job->confirmation_firstname.' '.$job->confirmation_lastname,
				]),
			]);

			if ($notified)
			{
				$notifiedUsers[] = $staff_id;

				// Send email notification
				$staff = $this->staff_model->get($staff_id);
				if ($staff && ! empty($staff->email))
				{
					$this->send_job_confirmation_email($staff->email, $staff_id, $job_id);
				}
			}
		}

		// Trigger pusher notifications
		pusher_trigger_notification($notifiedUsers);

		// Log the notification activity
		log_activity('Job Confirmation Notifications Sent [Job ID: '.$job_id.'] to '.count($notifiedUsers).' staff members');
	}

	/**
	 * Send email notification to staff about job confirmation
	 *
	 * @param string $staff_email Staff email address
	 * @param int $staff_id Staff ID
	 * @param int $job_id Job ID
	 * @return bool
	 */
	private function send_job_confirmation_email($staff_email, $staff_id, $job_id)
	{
		$path = APP_MODULES_PATH.TRADE_JOB_MANAGEMENT_MODULE_NAME.'/libraries/mails/Job_client_confirmed_to_staff.php';

		if ( ! file_exists($path))
		{
			return FALSE;
		}

		// Include the mailable class
		if ( ! class_exists('Job_client_confirmed_to_staff', FALSE))
		{
			include_once $path;
		}

		// Initialize and send email
		$instance = new Job_client_confirmed_to_staff($staff_email, $staff_id, $job_id);

		return $instance->send();
	}

	public function details($id = '', $hash = '')
	{
		check_job_restrictions($id, $hash);

		$job = $this->jobs_model->get($id);

		if ( ! is_client_logged_in() && isset($job->landlord))
		{
			load_client_language($job->landlord['client_id']);
		}

		// Load job confirmation helper
		$this->load->helper('trade_job_management/tjm_job_confirmation');

		$identity_confirmation_enabled = get_option('estimate_accept_identity_confirmation');

		// Handle job confirmation form submission
		if ($this->input->post('job_action'))
		{
			$action = $this->input->post('job_action');

			// Process job confirmation signature
			if ($action == 'signature')
			{
				$signature_data = $this->input->post('signature', FALSE);

				// Process and save signature image
				if ( ! empty($signature_data))
				{
					process_job_digital_signature($signature_data, $id);
				}

				// Update job with confirmation data
				$confirmation_data           = get_job_acceptance_info_array();

				$this->db->where('jobid', $id);
				$this->db->update(db_prefix().'tjm_jobs', $confirmation_data);

				// Log activity
				log_activity('Job Confirmed by Client [ID: '.$id.'] - '.$confirmation_data['confirmation_firstname'].' '.$confirmation_data['confirmation_lastname']);

				// Send notifications to staff
				$this->send_job_confirmation_notifications($id);

				// Show success message
				set_alert('success', _l('job_confirmed_successfully'));

				// Redirect back to view
				redirect(current_url());
			}
		}

		// Load sticky script
		$this->app_scripts->theme('sticky-js', 'assets/plugins/sticky/sticky.js');

		$this->disableNavigation();
		$this->disableSubMenu();

		$data['job']   = $job;
		$data['title'] = $job->job_title;

		// Get daily logs for customer view (only visible ones)
		$data['daily_logs'] = $this->job_daily_logs_model->get_by_job_id($id, TRUE);
		$data['statistics'] = $this->job_daily_logs_model->get_statistics($id);

		$data['hash']                          = $hash;
		$data['can_be_accepted']               = FALSE;
		$data['bodyclass']                     = 'viewestimate identity-confirmation';
		$data['identity_confirmation_enabled'] = $identity_confirmation_enabled;

		$data['use_navigation'] = FALSE;
		$data['use_submenu']    = FALSE;

		$data['navigationEnabled'] = FALSE;
		$data['subMenuEnabled']    = FALSE;
		$this->template['head']    = $this->load->view('themes/'.active_clients_theme().'/head', $data, TRUE);

		$module                 = CI::$APP->router->fetch_module();
		$data['current_module'] = $module;

		$this->title($job->job_title);
		$this->data($data);
		$this->view('jobs/job_details_html');
		$this->layout(TRUE);
	}

}
